﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/customer-profiles/CustomerProfilesRequest.h>
#include <aws/customer-profiles/CustomerProfiles_EXPORTS.h>
#include <aws/customer-profiles/model/AttributeDetails.h>
#include <aws/customer-profiles/model/Conditions.h>
#include <aws/customer-profiles/model/Filter.h>
#include <aws/customer-profiles/model/Statistic.h>

#include <utility>

namespace Aws {
namespace CustomerProfiles {
namespace Model {

/**
 */
class CreateCalculatedAttributeDefinitionRequest : public CustomerProfilesRequest {
 public:
  AWS_CUSTOMERPROFILES_API CreateCalculatedAttributeDefinitionRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateCalculatedAttributeDefinition"; }

  AWS_CUSTOMERPROFILES_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The unique name of the domain.</p>
   */
  inline const Aws::String& GetDomainName() const { return m_domainName; }
  inline bool DomainNameHasBeenSet() const { return m_domainNameHasBeenSet; }
  template <typename DomainNameT = Aws::String>
  void SetDomainName(DomainNameT&& value) {
    m_domainNameHasBeenSet = true;
    m_domainName = std::forward<DomainNameT>(value);
  }
  template <typename DomainNameT = Aws::String>
  CreateCalculatedAttributeDefinitionRequest& WithDomainName(DomainNameT&& value) {
    SetDomainName(std::forward<DomainNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The unique name of the calculated attribute.</p>
   */
  inline const Aws::String& GetCalculatedAttributeName() const { return m_calculatedAttributeName; }
  inline bool CalculatedAttributeNameHasBeenSet() const { return m_calculatedAttributeNameHasBeenSet; }
  template <typename CalculatedAttributeNameT = Aws::String>
  void SetCalculatedAttributeName(CalculatedAttributeNameT&& value) {
    m_calculatedAttributeNameHasBeenSet = true;
    m_calculatedAttributeName = std::forward<CalculatedAttributeNameT>(value);
  }
  template <typename CalculatedAttributeNameT = Aws::String>
  CreateCalculatedAttributeDefinitionRequest& WithCalculatedAttributeName(CalculatedAttributeNameT&& value) {
    SetCalculatedAttributeName(std::forward<CalculatedAttributeNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The display name of the calculated attribute.</p>
   */
  inline const Aws::String& GetDisplayName() const { return m_displayName; }
  inline bool DisplayNameHasBeenSet() const { return m_displayNameHasBeenSet; }
  template <typename DisplayNameT = Aws::String>
  void SetDisplayName(DisplayNameT&& value) {
    m_displayNameHasBeenSet = true;
    m_displayName = std::forward<DisplayNameT>(value);
  }
  template <typename DisplayNameT = Aws::String>
  CreateCalculatedAttributeDefinitionRequest& WithDisplayName(DisplayNameT&& value) {
    SetDisplayName(std::forward<DisplayNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The description of the calculated attribute.</p>
   */
  inline const Aws::String& GetDescription() const { return m_description; }
  inline bool DescriptionHasBeenSet() const { return m_descriptionHasBeenSet; }
  template <typename DescriptionT = Aws::String>
  void SetDescription(DescriptionT&& value) {
    m_descriptionHasBeenSet = true;
    m_description = std::forward<DescriptionT>(value);
  }
  template <typename DescriptionT = Aws::String>
  CreateCalculatedAttributeDefinitionRequest& WithDescription(DescriptionT&& value) {
    SetDescription(std::forward<DescriptionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Mathematical expression and a list of attribute items specified in that
   * expression.</p>
   */
  inline const AttributeDetails& GetAttributeDetails() const { return m_attributeDetails; }
  inline bool AttributeDetailsHasBeenSet() const { return m_attributeDetailsHasBeenSet; }
  template <typename AttributeDetailsT = AttributeDetails>
  void SetAttributeDetails(AttributeDetailsT&& value) {
    m_attributeDetailsHasBeenSet = true;
    m_attributeDetails = std::forward<AttributeDetailsT>(value);
  }
  template <typename AttributeDetailsT = AttributeDetails>
  CreateCalculatedAttributeDefinitionRequest& WithAttributeDetails(AttributeDetailsT&& value) {
    SetAttributeDetails(std::forward<AttributeDetailsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The conditions including range, object count, and threshold for the
   * calculated attribute.</p>
   */
  inline const Conditions& GetConditions() const { return m_conditions; }
  inline bool ConditionsHasBeenSet() const { return m_conditionsHasBeenSet; }
  template <typename ConditionsT = Conditions>
  void SetConditions(ConditionsT&& value) {
    m_conditionsHasBeenSet = true;
    m_conditions = std::forward<ConditionsT>(value);
  }
  template <typename ConditionsT = Conditions>
  CreateCalculatedAttributeDefinitionRequest& WithConditions(ConditionsT&& value) {
    SetConditions(std::forward<ConditionsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Defines how to filter incoming objects to include part of the Calculated
   * Attribute.</p>
   */
  inline const Filter& GetFilter() const { return m_filter; }
  inline bool FilterHasBeenSet() const { return m_filterHasBeenSet; }
  template <typename FilterT = Filter>
  void SetFilter(FilterT&& value) {
    m_filterHasBeenSet = true;
    m_filter = std::forward<FilterT>(value);
  }
  template <typename FilterT = Filter>
  CreateCalculatedAttributeDefinitionRequest& WithFilter(FilterT&& value) {
    SetFilter(std::forward<FilterT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The aggregation operation to perform for the calculated attribute.</p>
   */
  inline Statistic GetStatistic() const { return m_statistic; }
  inline bool StatisticHasBeenSet() const { return m_statisticHasBeenSet; }
  inline void SetStatistic(Statistic value) {
    m_statisticHasBeenSet = true;
    m_statistic = value;
  }
  inline CreateCalculatedAttributeDefinitionRequest& WithStatistic(Statistic value) {
    SetStatistic(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Whether historical data ingested before the Calculated Attribute was created
   * should be included in calculations.</p>
   */
  inline bool GetUseHistoricalData() const { return m_useHistoricalData; }
  inline bool UseHistoricalDataHasBeenSet() const { return m_useHistoricalDataHasBeenSet; }
  inline void SetUseHistoricalData(bool value) {
    m_useHistoricalDataHasBeenSet = true;
    m_useHistoricalData = value;
  }
  inline CreateCalculatedAttributeDefinitionRequest& WithUseHistoricalData(bool value) {
    SetUseHistoricalData(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The tags used to organize, track, or control access for this resource.</p>
   */
  inline const Aws::Map<Aws::String, Aws::String>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  CreateCalculatedAttributeDefinitionRequest& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsKeyT = Aws::String, typename TagsValueT = Aws::String>
  CreateCalculatedAttributeDefinitionRequest& AddTags(TagsKeyT&& key, TagsValueT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace(std::forward<TagsKeyT>(key), std::forward<TagsValueT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_domainName;
  bool m_domainNameHasBeenSet = false;

  Aws::String m_calculatedAttributeName;
  bool m_calculatedAttributeNameHasBeenSet = false;

  Aws::String m_displayName;
  bool m_displayNameHasBeenSet = false;

  Aws::String m_description;
  bool m_descriptionHasBeenSet = false;

  AttributeDetails m_attributeDetails;
  bool m_attributeDetailsHasBeenSet = false;

  Conditions m_conditions;
  bool m_conditionsHasBeenSet = false;

  Filter m_filter;
  bool m_filterHasBeenSet = false;

  Statistic m_statistic{Statistic::NOT_SET};
  bool m_statisticHasBeenSet = false;

  bool m_useHistoricalData{false};
  bool m_useHistoricalDataHasBeenSet = false;

  Aws::Map<Aws::String, Aws::String> m_tags;
  bool m_tagsHasBeenSet = false;
};

}  // namespace Model
}  // namespace CustomerProfiles
}  // namespace Aws
