﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/deadline/DeadlineRequest.h>
#include <aws/deadline/Deadline_EXPORTS.h>

#include <utility>

namespace Aws {
namespace deadline {
namespace Model {

/**
 */
class UpdateLimitRequest : public DeadlineRequest {
 public:
  AWS_DEADLINE_API UpdateLimitRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "UpdateLimit"; }

  AWS_DEADLINE_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The unique identifier of the farm that contains the limit.</p>
   */
  inline const Aws::String& GetFarmId() const { return m_farmId; }
  inline bool FarmIdHasBeenSet() const { return m_farmIdHasBeenSet; }
  template <typename FarmIdT = Aws::String>
  void SetFarmId(FarmIdT&& value) {
    m_farmIdHasBeenSet = true;
    m_farmId = std::forward<FarmIdT>(value);
  }
  template <typename FarmIdT = Aws::String>
  UpdateLimitRequest& WithFarmId(FarmIdT&& value) {
    SetFarmId(std::forward<FarmIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The unique identifier of the limit to update.</p>
   */
  inline const Aws::String& GetLimitId() const { return m_limitId; }
  inline bool LimitIdHasBeenSet() const { return m_limitIdHasBeenSet; }
  template <typename LimitIdT = Aws::String>
  void SetLimitId(LimitIdT&& value) {
    m_limitIdHasBeenSet = true;
    m_limitId = std::forward<LimitIdT>(value);
  }
  template <typename LimitIdT = Aws::String>
  UpdateLimitRequest& WithLimitId(LimitIdT&& value) {
    SetLimitId(std::forward<LimitIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The new display name of the limit.</p>  <p>This field can store
   * any content. Escape or encode this content before displaying it on a webpage or
   * any other system that might interpret the content of this field.</p>
   *
   */
  inline const Aws::String& GetDisplayName() const { return m_displayName; }
  inline bool DisplayNameHasBeenSet() const { return m_displayNameHasBeenSet; }
  template <typename DisplayNameT = Aws::String>
  void SetDisplayName(DisplayNameT&& value) {
    m_displayNameHasBeenSet = true;
    m_displayName = std::forward<DisplayNameT>(value);
  }
  template <typename DisplayNameT = Aws::String>
  UpdateLimitRequest& WithDisplayName(DisplayNameT&& value) {
    SetDisplayName(std::forward<DisplayNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The new description of the limit.</p>  <p>This field can store any
   * content. Escape or encode this content before displaying it on a webpage or any
   * other system that might interpret the content of this field.</p>
   */
  inline const Aws::String& GetDescription() const { return m_description; }
  inline bool DescriptionHasBeenSet() const { return m_descriptionHasBeenSet; }
  template <typename DescriptionT = Aws::String>
  void SetDescription(DescriptionT&& value) {
    m_descriptionHasBeenSet = true;
    m_description = std::forward<DescriptionT>(value);
  }
  template <typename DescriptionT = Aws::String>
  UpdateLimitRequest& WithDescription(DescriptionT&& value) {
    SetDescription(std::forward<DescriptionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The maximum number of resources constrained by this limit. When all of the
   * resources are in use, steps that require the limit won't be scheduled until the
   * resource is available.</p> <p>If more than the new maximum number is currently
   * in use, running jobs finish but no new jobs are started until the number of
   * resources in use is below the new maximum number.</p> <p>The
   * <code>maxCount</code> must not be 0. If the value is -1, there is no restriction
   * on the number of resources that can be acquired for this limit.</p>
   */
  inline int GetMaxCount() const { return m_maxCount; }
  inline bool MaxCountHasBeenSet() const { return m_maxCountHasBeenSet; }
  inline void SetMaxCount(int value) {
    m_maxCountHasBeenSet = true;
    m_maxCount = value;
  }
  inline UpdateLimitRequest& WithMaxCount(int value) {
    SetMaxCount(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_farmId;
  bool m_farmIdHasBeenSet = false;

  Aws::String m_limitId;
  bool m_limitIdHasBeenSet = false;

  Aws::String m_displayName;
  bool m_displayNameHasBeenSet = false;

  Aws::String m_description;
  bool m_descriptionHasBeenSet = false;

  int m_maxCount{0};
  bool m_maxCountHasBeenSet = false;
};

}  // namespace Model
}  // namespace deadline
}  // namespace Aws
