﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/DateTime.h>
#include <aws/core/utils/memory/stl/AWSStreamFwd.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/docdb/DocDB_EXPORTS.h>
#include <aws/docdb/model/SourceType.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Xml {
class XmlNode;
}  // namespace Xml
}  // namespace Utils
namespace DocDB {
namespace Model {

/**
 * <p>Detailed information about an event.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/docdb-2014-10-31/Event">AWS API
 * Reference</a></p>
 */
class Event {
 public:
  AWS_DOCDB_API Event() = default;
  AWS_DOCDB_API Event(const Aws::Utils::Xml::XmlNode& xmlNode);
  AWS_DOCDB_API Event& operator=(const Aws::Utils::Xml::XmlNode& xmlNode);

  AWS_DOCDB_API void OutputToStream(Aws::OStream& ostream, const char* location, unsigned index, const char* locationValue) const;
  AWS_DOCDB_API void OutputToStream(Aws::OStream& oStream, const char* location) const;

  ///@{
  /**
   * <p>Provides the identifier for the source of the event.</p>
   */
  inline const Aws::String& GetSourceIdentifier() const { return m_sourceIdentifier; }
  inline bool SourceIdentifierHasBeenSet() const { return m_sourceIdentifierHasBeenSet; }
  template <typename SourceIdentifierT = Aws::String>
  void SetSourceIdentifier(SourceIdentifierT&& value) {
    m_sourceIdentifierHasBeenSet = true;
    m_sourceIdentifier = std::forward<SourceIdentifierT>(value);
  }
  template <typename SourceIdentifierT = Aws::String>
  Event& WithSourceIdentifier(SourceIdentifierT&& value) {
    SetSourceIdentifier(std::forward<SourceIdentifierT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies the source type for this event.</p>
   */
  inline SourceType GetSourceType() const { return m_sourceType; }
  inline bool SourceTypeHasBeenSet() const { return m_sourceTypeHasBeenSet; }
  inline void SetSourceType(SourceType value) {
    m_sourceTypeHasBeenSet = true;
    m_sourceType = value;
  }
  inline Event& WithSourceType(SourceType value) {
    SetSourceType(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Provides the text of this event.</p>
   */
  inline const Aws::String& GetMessage() const { return m_message; }
  inline bool MessageHasBeenSet() const { return m_messageHasBeenSet; }
  template <typename MessageT = Aws::String>
  void SetMessage(MessageT&& value) {
    m_messageHasBeenSet = true;
    m_message = std::forward<MessageT>(value);
  }
  template <typename MessageT = Aws::String>
  Event& WithMessage(MessageT&& value) {
    SetMessage(std::forward<MessageT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies the category for the event.</p>
   */
  inline const Aws::Vector<Aws::String>& GetEventCategories() const { return m_eventCategories; }
  inline bool EventCategoriesHasBeenSet() const { return m_eventCategoriesHasBeenSet; }
  template <typename EventCategoriesT = Aws::Vector<Aws::String>>
  void SetEventCategories(EventCategoriesT&& value) {
    m_eventCategoriesHasBeenSet = true;
    m_eventCategories = std::forward<EventCategoriesT>(value);
  }
  template <typename EventCategoriesT = Aws::Vector<Aws::String>>
  Event& WithEventCategories(EventCategoriesT&& value) {
    SetEventCategories(std::forward<EventCategoriesT>(value));
    return *this;
  }
  template <typename EventCategoriesT = Aws::String>
  Event& AddEventCategories(EventCategoriesT&& value) {
    m_eventCategoriesHasBeenSet = true;
    m_eventCategories.emplace_back(std::forward<EventCategoriesT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies the date and time of the event.</p>
   */
  inline const Aws::Utils::DateTime& GetDate() const { return m_date; }
  inline bool DateHasBeenSet() const { return m_dateHasBeenSet; }
  template <typename DateT = Aws::Utils::DateTime>
  void SetDate(DateT&& value) {
    m_dateHasBeenSet = true;
    m_date = std::forward<DateT>(value);
  }
  template <typename DateT = Aws::Utils::DateTime>
  Event& WithDate(DateT&& value) {
    SetDate(std::forward<DateT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) for the event.</p>
   */
  inline const Aws::String& GetSourceArn() const { return m_sourceArn; }
  inline bool SourceArnHasBeenSet() const { return m_sourceArnHasBeenSet; }
  template <typename SourceArnT = Aws::String>
  void SetSourceArn(SourceArnT&& value) {
    m_sourceArnHasBeenSet = true;
    m_sourceArn = std::forward<SourceArnT>(value);
  }
  template <typename SourceArnT = Aws::String>
  Event& WithSourceArn(SourceArnT&& value) {
    SetSourceArn(std::forward<SourceArnT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_sourceIdentifier;
  bool m_sourceIdentifierHasBeenSet = false;

  SourceType m_sourceType{SourceType::NOT_SET};
  bool m_sourceTypeHasBeenSet = false;

  Aws::String m_message;
  bool m_messageHasBeenSet = false;

  Aws::Vector<Aws::String> m_eventCategories;
  bool m_eventCategoriesHasBeenSet = false;

  Aws::Utils::DateTime m_date{};
  bool m_dateHasBeenSet = false;

  Aws::String m_sourceArn;
  bool m_sourceArnHasBeenSet = false;
};

}  // namespace Model
}  // namespace DocDB
}  // namespace Aws
