﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/ecs/ECSRequest.h>
#include <aws/ecs/ECS_EXPORTS.h>

#include <utility>

namespace Aws {
namespace ECS {
namespace Model {

/**
 */
class StopTaskRequest : public ECSRequest {
 public:
  AWS_ECS_API StopTaskRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "StopTask"; }

  AWS_ECS_API Aws::String SerializePayload() const override;

  AWS_ECS_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The short name or full Amazon Resource Name (ARN) of the cluster that hosts
   * the task to stop. If you do not specify a cluster, the default cluster is
   * assumed.</p>
   */
  inline const Aws::String& GetCluster() const { return m_cluster; }
  inline bool ClusterHasBeenSet() const { return m_clusterHasBeenSet; }
  template <typename ClusterT = Aws::String>
  void SetCluster(ClusterT&& value) {
    m_clusterHasBeenSet = true;
    m_cluster = std::forward<ClusterT>(value);
  }
  template <typename ClusterT = Aws::String>
  StopTaskRequest& WithCluster(ClusterT&& value) {
    SetCluster(std::forward<ClusterT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Thefull Amazon Resource Name (ARN) of the task.</p>
   */
  inline const Aws::String& GetTask() const { return m_task; }
  inline bool TaskHasBeenSet() const { return m_taskHasBeenSet; }
  template <typename TaskT = Aws::String>
  void SetTask(TaskT&& value) {
    m_taskHasBeenSet = true;
    m_task = std::forward<TaskT>(value);
  }
  template <typename TaskT = Aws::String>
  StopTaskRequest& WithTask(TaskT&& value) {
    SetTask(std::forward<TaskT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>An optional message specified when a task is stopped. For example, if you're
   * using a custom scheduler, you can use this parameter to specify the reason for
   * stopping the task here, and the message appears in subsequent <a
   * href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_DescribeTasks.html">DescribeTasks</a>&gt;
   * API operations on this task.</p>
   */
  inline const Aws::String& GetReason() const { return m_reason; }
  inline bool ReasonHasBeenSet() const { return m_reasonHasBeenSet; }
  template <typename ReasonT = Aws::String>
  void SetReason(ReasonT&& value) {
    m_reasonHasBeenSet = true;
    m_reason = std::forward<ReasonT>(value);
  }
  template <typename ReasonT = Aws::String>
  StopTaskRequest& WithReason(ReasonT&& value) {
    SetReason(std::forward<ReasonT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_cluster;
  bool m_clusterHasBeenSet = false;

  Aws::String m_task;
  bool m_taskHasBeenSet = false;

  Aws::String m_reason;
  bool m_reasonHasBeenSet = false;
};

}  // namespace Model
}  // namespace ECS
}  // namespace Aws
