﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/UUID.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/eks/EKSRequest.h>
#include <aws/eks/EKS_EXPORTS.h>
#include <aws/eks/model/EncryptionConfig.h>

#include <utility>

namespace Aws {
namespace EKS {
namespace Model {

/**
 */
class AssociateEncryptionConfigRequest : public EKSRequest {
 public:
  AWS_EKS_API AssociateEncryptionConfigRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "AssociateEncryptionConfig"; }

  AWS_EKS_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The name of your cluster.</p>
   */
  inline const Aws::String& GetClusterName() const { return m_clusterName; }
  inline bool ClusterNameHasBeenSet() const { return m_clusterNameHasBeenSet; }
  template <typename ClusterNameT = Aws::String>
  void SetClusterName(ClusterNameT&& value) {
    m_clusterNameHasBeenSet = true;
    m_clusterName = std::forward<ClusterNameT>(value);
  }
  template <typename ClusterNameT = Aws::String>
  AssociateEncryptionConfigRequest& WithClusterName(ClusterNameT&& value) {
    SetClusterName(std::forward<ClusterNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The configuration you are using for encryption.</p>
   */
  inline const Aws::Vector<EncryptionConfig>& GetEncryptionConfig() const { return m_encryptionConfig; }
  inline bool EncryptionConfigHasBeenSet() const { return m_encryptionConfigHasBeenSet; }
  template <typename EncryptionConfigT = Aws::Vector<EncryptionConfig>>
  void SetEncryptionConfig(EncryptionConfigT&& value) {
    m_encryptionConfigHasBeenSet = true;
    m_encryptionConfig = std::forward<EncryptionConfigT>(value);
  }
  template <typename EncryptionConfigT = Aws::Vector<EncryptionConfig>>
  AssociateEncryptionConfigRequest& WithEncryptionConfig(EncryptionConfigT&& value) {
    SetEncryptionConfig(std::forward<EncryptionConfigT>(value));
    return *this;
  }
  template <typename EncryptionConfigT = EncryptionConfig>
  AssociateEncryptionConfigRequest& AddEncryptionConfig(EncryptionConfigT&& value) {
    m_encryptionConfigHasBeenSet = true;
    m_encryptionConfig.emplace_back(std::forward<EncryptionConfigT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A unique, case-sensitive identifier that you provide to ensure the
   * idempotency of the request.</p>
   */
  inline const Aws::String& GetClientRequestToken() const { return m_clientRequestToken; }
  inline bool ClientRequestTokenHasBeenSet() const { return m_clientRequestTokenHasBeenSet; }
  template <typename ClientRequestTokenT = Aws::String>
  void SetClientRequestToken(ClientRequestTokenT&& value) {
    m_clientRequestTokenHasBeenSet = true;
    m_clientRequestToken = std::forward<ClientRequestTokenT>(value);
  }
  template <typename ClientRequestTokenT = Aws::String>
  AssociateEncryptionConfigRequest& WithClientRequestToken(ClientRequestTokenT&& value) {
    SetClientRequestToken(std::forward<ClientRequestTokenT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_clusterName;
  bool m_clusterNameHasBeenSet = false;

  Aws::Vector<EncryptionConfig> m_encryptionConfig;
  bool m_encryptionConfigHasBeenSet = false;

  Aws::String m_clientRequestToken{Aws::Utils::UUID::PseudoRandomUUID()};
  bool m_clientRequestTokenHasBeenSet = true;
};

}  // namespace Model
}  // namespace EKS
}  // namespace Aws
