﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/forecast/ForecastService_EXPORTS.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Json {
class JsonValue;
class JsonView;
}  // namespace Json
}  // namespace Utils
namespace ForecastService {
namespace Model {

/**
 * <p>The path to the file(s) in an Amazon Simple Storage Service (Amazon S3)
 * bucket, and an Identity and Access Management (IAM) role that Amazon Forecast
 * can assume to access the file(s). Optionally, includes an Key Management Service
 * (KMS) key. This object is part of the <a>DataSource</a> object that is submitted
 * in the <a>CreateDatasetImportJob</a> request, and part of the
 * <a>DataDestination</a> object.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/forecast-2018-06-26/S3Config">AWS
 * API Reference</a></p>
 */
class S3Config {
 public:
  AWS_FORECASTSERVICE_API S3Config() = default;
  AWS_FORECASTSERVICE_API S3Config(Aws::Utils::Json::JsonView jsonValue);
  AWS_FORECASTSERVICE_API S3Config& operator=(Aws::Utils::Json::JsonView jsonValue);
  AWS_FORECASTSERVICE_API Aws::Utils::Json::JsonValue Jsonize() const;

  ///@{
  /**
   * <p>The path to an Amazon Simple Storage Service (Amazon S3) bucket or file(s) in
   * an Amazon S3 bucket.</p>
   */
  inline const Aws::String& GetPath() const { return m_path; }
  inline bool PathHasBeenSet() const { return m_pathHasBeenSet; }
  template <typename PathT = Aws::String>
  void SetPath(PathT&& value) {
    m_pathHasBeenSet = true;
    m_path = std::forward<PathT>(value);
  }
  template <typename PathT = Aws::String>
  S3Config& WithPath(PathT&& value) {
    SetPath(std::forward<PathT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ARN of the Identity and Access Management (IAM) role that Amazon Forecast
   * can assume to access the Amazon S3 bucket or files. If you provide a value for
   * the <code>KMSKeyArn</code> key, the role must allow access to the key.</p>
   * <p>Passing a role across Amazon Web Services accounts is not allowed. If you
   * pass a role that isn't in your account, you get an
   * <code>InvalidInputException</code> error.</p>
   */
  inline const Aws::String& GetRoleArn() const { return m_roleArn; }
  inline bool RoleArnHasBeenSet() const { return m_roleArnHasBeenSet; }
  template <typename RoleArnT = Aws::String>
  void SetRoleArn(RoleArnT&& value) {
    m_roleArnHasBeenSet = true;
    m_roleArn = std::forward<RoleArnT>(value);
  }
  template <typename RoleArnT = Aws::String>
  S3Config& WithRoleArn(RoleArnT&& value) {
    SetRoleArn(std::forward<RoleArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of an Key Management Service (KMS) key.</p>
   */
  inline const Aws::String& GetKMSKeyArn() const { return m_kMSKeyArn; }
  inline bool KMSKeyArnHasBeenSet() const { return m_kMSKeyArnHasBeenSet; }
  template <typename KMSKeyArnT = Aws::String>
  void SetKMSKeyArn(KMSKeyArnT&& value) {
    m_kMSKeyArnHasBeenSet = true;
    m_kMSKeyArn = std::forward<KMSKeyArnT>(value);
  }
  template <typename KMSKeyArnT = Aws::String>
  S3Config& WithKMSKeyArn(KMSKeyArnT&& value) {
    SetKMSKeyArn(std::forward<KMSKeyArnT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_path;
  bool m_pathHasBeenSet = false;

  Aws::String m_roleArn;
  bool m_roleArnHasBeenSet = false;

  Aws::String m_kMSKeyArn;
  bool m_kMSKeyArnHasBeenSet = false;
};

}  // namespace Model
}  // namespace ForecastService
}  // namespace Aws
