﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/UUID.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/fsx/FSxRequest.h>
#include <aws/fsx/FSx_EXPORTS.h>
#include <aws/fsx/model/UpdateFileCacheLustreConfiguration.h>

#include <utility>

namespace Aws {
namespace FSx {
namespace Model {

/**
 */
class UpdateFileCacheRequest : public FSxRequest {
 public:
  AWS_FSX_API UpdateFileCacheRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "UpdateFileCache"; }

  AWS_FSX_API Aws::String SerializePayload() const override;

  AWS_FSX_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The ID of the cache that you are updating.</p>
   */
  inline const Aws::String& GetFileCacheId() const { return m_fileCacheId; }
  inline bool FileCacheIdHasBeenSet() const { return m_fileCacheIdHasBeenSet; }
  template <typename FileCacheIdT = Aws::String>
  void SetFileCacheId(FileCacheIdT&& value) {
    m_fileCacheIdHasBeenSet = true;
    m_fileCacheId = std::forward<FileCacheIdT>(value);
  }
  template <typename FileCacheIdT = Aws::String>
  UpdateFileCacheRequest& WithFileCacheId(FileCacheIdT&& value) {
    SetFileCacheId(std::forward<FileCacheIdT>(value));
    return *this;
  }
  ///@}

  ///@{

  inline const Aws::String& GetClientRequestToken() const { return m_clientRequestToken; }
  inline bool ClientRequestTokenHasBeenSet() const { return m_clientRequestTokenHasBeenSet; }
  template <typename ClientRequestTokenT = Aws::String>
  void SetClientRequestToken(ClientRequestTokenT&& value) {
    m_clientRequestTokenHasBeenSet = true;
    m_clientRequestToken = std::forward<ClientRequestTokenT>(value);
  }
  template <typename ClientRequestTokenT = Aws::String>
  UpdateFileCacheRequest& WithClientRequestToken(ClientRequestTokenT&& value) {
    SetClientRequestToken(std::forward<ClientRequestTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The configuration updates for an Amazon File Cache resource.</p>
   */
  inline const UpdateFileCacheLustreConfiguration& GetLustreConfiguration() const { return m_lustreConfiguration; }
  inline bool LustreConfigurationHasBeenSet() const { return m_lustreConfigurationHasBeenSet; }
  template <typename LustreConfigurationT = UpdateFileCacheLustreConfiguration>
  void SetLustreConfiguration(LustreConfigurationT&& value) {
    m_lustreConfigurationHasBeenSet = true;
    m_lustreConfiguration = std::forward<LustreConfigurationT>(value);
  }
  template <typename LustreConfigurationT = UpdateFileCacheLustreConfiguration>
  UpdateFileCacheRequest& WithLustreConfiguration(LustreConfigurationT&& value) {
    SetLustreConfiguration(std::forward<LustreConfigurationT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_fileCacheId;
  bool m_fileCacheIdHasBeenSet = false;

  Aws::String m_clientRequestToken{Aws::Utils::UUID::PseudoRandomUUID()};
  bool m_clientRequestTokenHasBeenSet = true;

  UpdateFileCacheLustreConfiguration m_lustreConfiguration;
  bool m_lustreConfigurationHasBeenSet = false;
};

}  // namespace Model
}  // namespace FSx
}  // namespace Aws
