﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/glue/Glue_EXPORTS.h>
#include <aws/glue/model/NotificationProperty.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Json {
class JsonValue;
class JsonView;
}  // namespace Json
}  // namespace Utils
namespace Glue {
namespace Model {

/**
 * <p>Defines an action to be initiated by a trigger.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/glue-2017-03-31/Action">AWS API
 * Reference</a></p>
 */
class Action {
 public:
  AWS_GLUE_API Action() = default;
  AWS_GLUE_API Action(Aws::Utils::Json::JsonView jsonValue);
  AWS_GLUE_API Action& operator=(Aws::Utils::Json::JsonView jsonValue);
  AWS_GLUE_API Aws::Utils::Json::JsonValue Jsonize() const;

  ///@{
  /**
   * <p>The name of a job to be run.</p>
   */
  inline const Aws::String& GetJobName() const { return m_jobName; }
  inline bool JobNameHasBeenSet() const { return m_jobNameHasBeenSet; }
  template <typename JobNameT = Aws::String>
  void SetJobName(JobNameT&& value) {
    m_jobNameHasBeenSet = true;
    m_jobName = std::forward<JobNameT>(value);
  }
  template <typename JobNameT = Aws::String>
  Action& WithJobName(JobNameT&& value) {
    SetJobName(std::forward<JobNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The job arguments used when this trigger fires. For this job run, they
   * replace the default arguments set in the job definition itself.</p> <p>You can
   * specify arguments here that your own job-execution script consumes, as well as
   * arguments that Glue itself consumes.</p> <p>For information about how to specify
   * and consume your own Job arguments, see the <a
   * href="https://docs.aws.amazon.com/glue/latest/dg/aws-glue-programming-python-calling.html">Calling
   * Glue APIs in Python</a> topic in the developer guide.</p> <p>For information
   * about the key-value pairs that Glue consumes to set up your job, see the <a
   * href="https://docs.aws.amazon.com/glue/latest/dg/aws-glue-programming-etl-glue-arguments.html">Special
   * Parameters Used by Glue</a> topic in the developer guide.</p>
   */
  inline const Aws::Map<Aws::String, Aws::String>& GetArguments() const { return m_arguments; }
  inline bool ArgumentsHasBeenSet() const { return m_argumentsHasBeenSet; }
  template <typename ArgumentsT = Aws::Map<Aws::String, Aws::String>>
  void SetArguments(ArgumentsT&& value) {
    m_argumentsHasBeenSet = true;
    m_arguments = std::forward<ArgumentsT>(value);
  }
  template <typename ArgumentsT = Aws::Map<Aws::String, Aws::String>>
  Action& WithArguments(ArgumentsT&& value) {
    SetArguments(std::forward<ArgumentsT>(value));
    return *this;
  }
  template <typename ArgumentsKeyT = Aws::String, typename ArgumentsValueT = Aws::String>
  Action& AddArguments(ArgumentsKeyT&& key, ArgumentsValueT&& value) {
    m_argumentsHasBeenSet = true;
    m_arguments.emplace(std::forward<ArgumentsKeyT>(key), std::forward<ArgumentsValueT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The <code>JobRun</code> timeout in minutes. This is the maximum time that a
   * job run can consume resources before it is terminated and enters
   * <code>TIMEOUT</code> status. This overrides the timeout value set in the parent
   * job.</p> <p>Jobs must have timeout values less than 7 days or 10080 minutes.
   * Otherwise, the jobs will throw an exception.</p> <p>When the value is left
   * blank, the timeout is defaulted to 2880 minutes.</p> <p>Any existing Glue jobs
   * that had a timeout value greater than 7 days will be defaulted to 7 days. For
   * instance if you have specified a timeout of 20 days for a batch job, it will be
   * stopped on the 7th day.</p> <p>For streaming jobs, if you have set up a
   * maintenance window, it will be restarted during the maintenance window after 7
   * days.</p>
   */
  inline int GetTimeout() const { return m_timeout; }
  inline bool TimeoutHasBeenSet() const { return m_timeoutHasBeenSet; }
  inline void SetTimeout(int value) {
    m_timeoutHasBeenSet = true;
    m_timeout = value;
  }
  inline Action& WithTimeout(int value) {
    SetTimeout(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the <code>SecurityConfiguration</code> structure to be used with
   * this action.</p>
   */
  inline const Aws::String& GetSecurityConfiguration() const { return m_securityConfiguration; }
  inline bool SecurityConfigurationHasBeenSet() const { return m_securityConfigurationHasBeenSet; }
  template <typename SecurityConfigurationT = Aws::String>
  void SetSecurityConfiguration(SecurityConfigurationT&& value) {
    m_securityConfigurationHasBeenSet = true;
    m_securityConfiguration = std::forward<SecurityConfigurationT>(value);
  }
  template <typename SecurityConfigurationT = Aws::String>
  Action& WithSecurityConfiguration(SecurityConfigurationT&& value) {
    SetSecurityConfiguration(std::forward<SecurityConfigurationT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies configuration properties of a job run notification.</p>
   */
  inline const NotificationProperty& GetNotificationProperty() const { return m_notificationProperty; }
  inline bool NotificationPropertyHasBeenSet() const { return m_notificationPropertyHasBeenSet; }
  template <typename NotificationPropertyT = NotificationProperty>
  void SetNotificationProperty(NotificationPropertyT&& value) {
    m_notificationPropertyHasBeenSet = true;
    m_notificationProperty = std::forward<NotificationPropertyT>(value);
  }
  template <typename NotificationPropertyT = NotificationProperty>
  Action& WithNotificationProperty(NotificationPropertyT&& value) {
    SetNotificationProperty(std::forward<NotificationPropertyT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the crawler to be used with this action.</p>
   */
  inline const Aws::String& GetCrawlerName() const { return m_crawlerName; }
  inline bool CrawlerNameHasBeenSet() const { return m_crawlerNameHasBeenSet; }
  template <typename CrawlerNameT = Aws::String>
  void SetCrawlerName(CrawlerNameT&& value) {
    m_crawlerNameHasBeenSet = true;
    m_crawlerName = std::forward<CrawlerNameT>(value);
  }
  template <typename CrawlerNameT = Aws::String>
  Action& WithCrawlerName(CrawlerNameT&& value) {
    SetCrawlerName(std::forward<CrawlerNameT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_jobName;
  bool m_jobNameHasBeenSet = false;

  Aws::Map<Aws::String, Aws::String> m_arguments;
  bool m_argumentsHasBeenSet = false;

  int m_timeout{0};
  bool m_timeoutHasBeenSet = false;

  Aws::String m_securityConfiguration;
  bool m_securityConfigurationHasBeenSet = false;

  NotificationProperty m_notificationProperty;
  bool m_notificationPropertyHasBeenSet = false;

  Aws::String m_crawlerName;
  bool m_crawlerNameHasBeenSet = false;
};

}  // namespace Model
}  // namespace Glue
}  // namespace Aws
