﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/glue/GlueRequest.h>
#include <aws/glue/Glue_EXPORTS.h>
#include <aws/glue/model/InclusionAnnotationValue.h>

#include <utility>

namespace Aws {
namespace Glue {
namespace Model {

/**
 */
class PutDataQualityProfileAnnotationRequest : public GlueRequest {
 public:
  AWS_GLUE_API PutDataQualityProfileAnnotationRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "PutDataQualityProfileAnnotation"; }

  AWS_GLUE_API Aws::String SerializePayload() const override;

  AWS_GLUE_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The ID of the data quality monitoring profile to annotate.</p>
   */
  inline const Aws::String& GetProfileId() const { return m_profileId; }
  inline bool ProfileIdHasBeenSet() const { return m_profileIdHasBeenSet; }
  template <typename ProfileIdT = Aws::String>
  void SetProfileId(ProfileIdT&& value) {
    m_profileIdHasBeenSet = true;
    m_profileId = std::forward<ProfileIdT>(value);
  }
  template <typename ProfileIdT = Aws::String>
  PutDataQualityProfileAnnotationRequest& WithProfileId(ProfileIdT&& value) {
    SetProfileId(std::forward<ProfileIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The inclusion annotation value to apply to the profile.</p>
   */
  inline InclusionAnnotationValue GetInclusionAnnotation() const { return m_inclusionAnnotation; }
  inline bool InclusionAnnotationHasBeenSet() const { return m_inclusionAnnotationHasBeenSet; }
  inline void SetInclusionAnnotation(InclusionAnnotationValue value) {
    m_inclusionAnnotationHasBeenSet = true;
    m_inclusionAnnotation = value;
  }
  inline PutDataQualityProfileAnnotationRequest& WithInclusionAnnotation(InclusionAnnotationValue value) {
    SetInclusionAnnotation(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_profileId;
  bool m_profileIdHasBeenSet = false;

  InclusionAnnotationValue m_inclusionAnnotation{InclusionAnnotationValue::NOT_SET};
  bool m_inclusionAnnotationHasBeenSet = false;
};

}  // namespace Model
}  // namespace Glue
}  // namespace Aws
