﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/guardduty/GuardDutyRequest.h>
#include <aws/guardduty/GuardDuty_EXPORTS.h>

#include <utility>

namespace Aws {
namespace GuardDuty {
namespace Model {

/**
 */
class DeleteFilterRequest : public GuardDutyRequest {
 public:
  AWS_GUARDDUTY_API DeleteFilterRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "DeleteFilter"; }

  AWS_GUARDDUTY_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The unique ID of the detector that is associated with the filter.</p> <p>To
   * find the <code>detectorId</code> in the current Region, see the Settings page in
   * the GuardDuty console, or run the <a
   * href="https://docs.aws.amazon.com/guardduty/latest/APIReference/API_ListDetectors.html">ListDetectors</a>
   * API.</p>
   */
  inline const Aws::String& GetDetectorId() const { return m_detectorId; }
  inline bool DetectorIdHasBeenSet() const { return m_detectorIdHasBeenSet; }
  template <typename DetectorIdT = Aws::String>
  void SetDetectorId(DetectorIdT&& value) {
    m_detectorIdHasBeenSet = true;
    m_detectorId = std::forward<DetectorIdT>(value);
  }
  template <typename DetectorIdT = Aws::String>
  DeleteFilterRequest& WithDetectorId(DetectorIdT&& value) {
    SetDetectorId(std::forward<DetectorIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the filter that you want to delete.</p>
   */
  inline const Aws::String& GetFilterName() const { return m_filterName; }
  inline bool FilterNameHasBeenSet() const { return m_filterNameHasBeenSet; }
  template <typename FilterNameT = Aws::String>
  void SetFilterName(FilterNameT&& value) {
    m_filterNameHasBeenSet = true;
    m_filterName = std::forward<FilterNameT>(value);
  }
  template <typename FilterNameT = Aws::String>
  DeleteFilterRequest& WithFilterName(FilterNameT&& value) {
    SetFilterName(std::forward<FilterNameT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_detectorId;
  bool m_detectorIdHasBeenSet = false;

  Aws::String m_filterName;
  bool m_filterNameHasBeenSet = false;
};

}  // namespace Model
}  // namespace GuardDuty
}  // namespace Aws
