﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/UUID.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/imagebuilder/ImagebuilderRequest.h>
#include <aws/imagebuilder/Imagebuilder_EXPORTS.h>
#include <aws/imagebuilder/model/WorkflowStepActionType.h>

#include <utility>

namespace Aws {
namespace imagebuilder {
namespace Model {

/**
 */
class SendWorkflowStepActionRequest : public ImagebuilderRequest {
 public:
  AWS_IMAGEBUILDER_API SendWorkflowStepActionRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "SendWorkflowStepAction"; }

  AWS_IMAGEBUILDER_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>Uniquely identifies the workflow step that sent the step action.</p>
   */
  inline const Aws::String& GetStepExecutionId() const { return m_stepExecutionId; }
  inline bool StepExecutionIdHasBeenSet() const { return m_stepExecutionIdHasBeenSet; }
  template <typename StepExecutionIdT = Aws::String>
  void SetStepExecutionId(StepExecutionIdT&& value) {
    m_stepExecutionIdHasBeenSet = true;
    m_stepExecutionId = std::forward<StepExecutionIdT>(value);
  }
  template <typename StepExecutionIdT = Aws::String>
  SendWorkflowStepActionRequest& WithStepExecutionId(StepExecutionIdT&& value) {
    SetStepExecutionId(std::forward<StepExecutionIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the image build version to send action
   * for.</p>
   */
  inline const Aws::String& GetImageBuildVersionArn() const { return m_imageBuildVersionArn; }
  inline bool ImageBuildVersionArnHasBeenSet() const { return m_imageBuildVersionArnHasBeenSet; }
  template <typename ImageBuildVersionArnT = Aws::String>
  void SetImageBuildVersionArn(ImageBuildVersionArnT&& value) {
    m_imageBuildVersionArnHasBeenSet = true;
    m_imageBuildVersionArn = std::forward<ImageBuildVersionArnT>(value);
  }
  template <typename ImageBuildVersionArnT = Aws::String>
  SendWorkflowStepActionRequest& WithImageBuildVersionArn(ImageBuildVersionArnT&& value) {
    SetImageBuildVersionArn(std::forward<ImageBuildVersionArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The action for the image creation process to take while a workflow
   * <code>WaitForAction</code> step waits for an asynchronous action to
   * complete.</p>
   */
  inline WorkflowStepActionType GetAction() const { return m_action; }
  inline bool ActionHasBeenSet() const { return m_actionHasBeenSet; }
  inline void SetAction(WorkflowStepActionType value) {
    m_actionHasBeenSet = true;
    m_action = value;
  }
  inline SendWorkflowStepActionRequest& WithAction(WorkflowStepActionType value) {
    SetAction(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The reason why this action is sent.</p>
   */
  inline const Aws::String& GetReason() const { return m_reason; }
  inline bool ReasonHasBeenSet() const { return m_reasonHasBeenSet; }
  template <typename ReasonT = Aws::String>
  void SetReason(ReasonT&& value) {
    m_reasonHasBeenSet = true;
    m_reason = std::forward<ReasonT>(value);
  }
  template <typename ReasonT = Aws::String>
  SendWorkflowStepActionRequest& WithReason(ReasonT&& value) {
    SetReason(std::forward<ReasonT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Unique, case-sensitive identifier you provide to ensure idempotency of the
   * request. For more information, see <a
   * href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/Run_Instance_Idempotency.html">Ensuring
   * idempotency</a> in the <i>Amazon EC2 API Reference</i>.</p>
   */
  inline const Aws::String& GetClientToken() const { return m_clientToken; }
  inline bool ClientTokenHasBeenSet() const { return m_clientTokenHasBeenSet; }
  template <typename ClientTokenT = Aws::String>
  void SetClientToken(ClientTokenT&& value) {
    m_clientTokenHasBeenSet = true;
    m_clientToken = std::forward<ClientTokenT>(value);
  }
  template <typename ClientTokenT = Aws::String>
  SendWorkflowStepActionRequest& WithClientToken(ClientTokenT&& value) {
    SetClientToken(std::forward<ClientTokenT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_stepExecutionId;
  bool m_stepExecutionIdHasBeenSet = false;

  Aws::String m_imageBuildVersionArn;
  bool m_imageBuildVersionArnHasBeenSet = false;

  WorkflowStepActionType m_action{WorkflowStepActionType::NOT_SET};
  bool m_actionHasBeenSet = false;

  Aws::String m_reason;
  bool m_reasonHasBeenSet = false;

  Aws::String m_clientToken{Aws::Utils::UUID::PseudoRandomUUID()};
  bool m_clientTokenHasBeenSet = true;
};

}  // namespace Model
}  // namespace imagebuilder
}  // namespace Aws
