﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/iot/IoTRequest.h>
#include <aws/iot/IoT_EXPORTS.h>

#include <utility>

namespace Aws {
namespace Http {
class URI;
}  // namespace Http
namespace IoT {
namespace Model {

/**
 */
class CancelJobRequest : public IoTRequest {
 public:
  AWS_IOT_API CancelJobRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CancelJob"; }

  AWS_IOT_API Aws::String SerializePayload() const override;

  AWS_IOT_API void AddQueryStringParameters(Aws::Http::URI& uri) const override;

  ///@{
  /**
   * <p>The unique identifier you assigned to this job when it was created.</p>
   */
  inline const Aws::String& GetJobId() const { return m_jobId; }
  inline bool JobIdHasBeenSet() const { return m_jobIdHasBeenSet; }
  template <typename JobIdT = Aws::String>
  void SetJobId(JobIdT&& value) {
    m_jobIdHasBeenSet = true;
    m_jobId = std::forward<JobIdT>(value);
  }
  template <typename JobIdT = Aws::String>
  CancelJobRequest& WithJobId(JobIdT&& value) {
    SetJobId(std::forward<JobIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>(Optional)A reason code string that explains why the job was canceled.</p>
   */
  inline const Aws::String& GetReasonCode() const { return m_reasonCode; }
  inline bool ReasonCodeHasBeenSet() const { return m_reasonCodeHasBeenSet; }
  template <typename ReasonCodeT = Aws::String>
  void SetReasonCode(ReasonCodeT&& value) {
    m_reasonCodeHasBeenSet = true;
    m_reasonCode = std::forward<ReasonCodeT>(value);
  }
  template <typename ReasonCodeT = Aws::String>
  CancelJobRequest& WithReasonCode(ReasonCodeT&& value) {
    SetReasonCode(std::forward<ReasonCodeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>An optional comment string describing why the job was canceled.</p>
   */
  inline const Aws::String& GetComment() const { return m_comment; }
  inline bool CommentHasBeenSet() const { return m_commentHasBeenSet; }
  template <typename CommentT = Aws::String>
  void SetComment(CommentT&& value) {
    m_commentHasBeenSet = true;
    m_comment = std::forward<CommentT>(value);
  }
  template <typename CommentT = Aws::String>
  CancelJobRequest& WithComment(CommentT&& value) {
    SetComment(std::forward<CommentT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>(Optional) If <code>true</code> job executions with status "IN_PROGRESS" and
   * "QUEUED" are canceled, otherwise only job executions with status "QUEUED" are
   * canceled. The default is <code>false</code>.</p> <p>Canceling a job which is
   * "IN_PROGRESS", will cause a device which is executing the job to be unable to
   * update the job execution status. Use caution and ensure that each device
   * executing a job which is canceled is able to recover to a valid state.</p>
   */
  inline bool GetForce() const { return m_force; }
  inline bool ForceHasBeenSet() const { return m_forceHasBeenSet; }
  inline void SetForce(bool value) {
    m_forceHasBeenSet = true;
    m_force = value;
  }
  inline CancelJobRequest& WithForce(bool value) {
    SetForce(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_jobId;
  bool m_jobIdHasBeenSet = false;

  Aws::String m_reasonCode;
  bool m_reasonCodeHasBeenSet = false;

  Aws::String m_comment;
  bool m_commentHasBeenSet = false;

  bool m_force{false};
  bool m_forceHasBeenSet = false;
};

}  // namespace Model
}  // namespace IoT
}  // namespace Aws
