﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/iotwireless/IoTWirelessRequest.h>
#include <aws/iotwireless/IoTWireless_EXPORTS.h>
#include <aws/iotwireless/model/WirelessDeviceType.h>

#include <utility>

namespace Aws {
namespace Http {
class URI;
}  // namespace Http
namespace IoTWireless {
namespace Model {

/**
 */
class DeregisterWirelessDeviceRequest : public IoTWirelessRequest {
 public:
  AWS_IOTWIRELESS_API DeregisterWirelessDeviceRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "DeregisterWirelessDevice"; }

  AWS_IOTWIRELESS_API Aws::String SerializePayload() const override;

  AWS_IOTWIRELESS_API void AddQueryStringParameters(Aws::Http::URI& uri) const override;

  ///@{
  /**
   * <p>The identifier of the wireless device to deregister from AWS IoT
   * Wireless.</p>
   */
  inline const Aws::String& GetIdentifier() const { return m_identifier; }
  inline bool IdentifierHasBeenSet() const { return m_identifierHasBeenSet; }
  template <typename IdentifierT = Aws::String>
  void SetIdentifier(IdentifierT&& value) {
    m_identifierHasBeenSet = true;
    m_identifier = std::forward<IdentifierT>(value);
  }
  template <typename IdentifierT = Aws::String>
  DeregisterWirelessDeviceRequest& WithIdentifier(IdentifierT&& value) {
    SetIdentifier(std::forward<IdentifierT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The type of wireless device to deregister from AWS IoT Wireless, which can be
   * <code>LoRaWAN</code> or <code>Sidewalk</code>.</p>
   */
  inline WirelessDeviceType GetWirelessDeviceType() const { return m_wirelessDeviceType; }
  inline bool WirelessDeviceTypeHasBeenSet() const { return m_wirelessDeviceTypeHasBeenSet; }
  inline void SetWirelessDeviceType(WirelessDeviceType value) {
    m_wirelessDeviceTypeHasBeenSet = true;
    m_wirelessDeviceType = value;
  }
  inline DeregisterWirelessDeviceRequest& WithWirelessDeviceType(WirelessDeviceType value) {
    SetWirelessDeviceType(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_identifier;
  bool m_identifierHasBeenSet = false;

  WirelessDeviceType m_wirelessDeviceType{WirelessDeviceType::NOT_SET};
  bool m_wirelessDeviceTypeHasBeenSet = false;
};

}  // namespace Model
}  // namespace IoTWireless
}  // namespace Aws
