﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/lexv2-models/LexModelsV2Request.h>
#include <aws/lexv2-models/LexModelsV2_EXPORTS.h>
#include <aws/lexv2-models/model/GenerativeAISettings.h>
#include <aws/lexv2-models/model/SpeechDetectionSensitivity.h>
#include <aws/lexv2-models/model/SpeechRecognitionSettings.h>
#include <aws/lexv2-models/model/UnifiedSpeechSettings.h>
#include <aws/lexv2-models/model/VoiceSettings.h>

#include <utility>

namespace Aws {
namespace LexModelsV2 {
namespace Model {

/**
 */
class CreateBotLocaleRequest : public LexModelsV2Request {
 public:
  AWS_LEXMODELSV2_API CreateBotLocaleRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateBotLocale"; }

  AWS_LEXMODELSV2_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The identifier of the bot to create the locale for.</p>
   */
  inline const Aws::String& GetBotId() const { return m_botId; }
  inline bool BotIdHasBeenSet() const { return m_botIdHasBeenSet; }
  template <typename BotIdT = Aws::String>
  void SetBotId(BotIdT&& value) {
    m_botIdHasBeenSet = true;
    m_botId = std::forward<BotIdT>(value);
  }
  template <typename BotIdT = Aws::String>
  CreateBotLocaleRequest& WithBotId(BotIdT&& value) {
    SetBotId(std::forward<BotIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The version of the bot to create the locale for. This can only be the draft
   * version of the bot.</p>
   */
  inline const Aws::String& GetBotVersion() const { return m_botVersion; }
  inline bool BotVersionHasBeenSet() const { return m_botVersionHasBeenSet; }
  template <typename BotVersionT = Aws::String>
  void SetBotVersion(BotVersionT&& value) {
    m_botVersionHasBeenSet = true;
    m_botVersion = std::forward<BotVersionT>(value);
  }
  template <typename BotVersionT = Aws::String>
  CreateBotLocaleRequest& WithBotVersion(BotVersionT&& value) {
    SetBotVersion(std::forward<BotVersionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The identifier of the language and locale that the bot will be used in. The
   * string must match one of the supported locales. All of the intents, slot types,
   * and slots used in the bot must have the same locale. For more information, see
   * <a
   * href="https://docs.aws.amazon.com/lexv2/latest/dg/how-languages.html">Supported
   * languages</a>.</p>
   */
  inline const Aws::String& GetLocaleId() const { return m_localeId; }
  inline bool LocaleIdHasBeenSet() const { return m_localeIdHasBeenSet; }
  template <typename LocaleIdT = Aws::String>
  void SetLocaleId(LocaleIdT&& value) {
    m_localeIdHasBeenSet = true;
    m_localeId = std::forward<LocaleIdT>(value);
  }
  template <typename LocaleIdT = Aws::String>
  CreateBotLocaleRequest& WithLocaleId(LocaleIdT&& value) {
    SetLocaleId(std::forward<LocaleIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A description of the bot locale. Use this to help identify the bot locale in
   * lists.</p>
   */
  inline const Aws::String& GetDescription() const { return m_description; }
  inline bool DescriptionHasBeenSet() const { return m_descriptionHasBeenSet; }
  template <typename DescriptionT = Aws::String>
  void SetDescription(DescriptionT&& value) {
    m_descriptionHasBeenSet = true;
    m_description = std::forward<DescriptionT>(value);
  }
  template <typename DescriptionT = Aws::String>
  CreateBotLocaleRequest& WithDescription(DescriptionT&& value) {
    SetDescription(std::forward<DescriptionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Determines the threshold where Amazon Lex will insert the
   * <code>AMAZON.FallbackIntent</code>, <code>AMAZON.KendraSearchIntent</code>, or
   * both when returning alternative intents. <code>AMAZON.FallbackIntent</code> and
   * <code>AMAZON.KendraSearchIntent</code> are only inserted if they are configured
   * for the bot.</p> <p>For example, suppose a bot is configured with the confidence
   * threshold of 0.80 and the <code>AMAZON.FallbackIntent</code>. Amazon Lex returns
   * three alternative intents with the following confidence scores: IntentA (0.70),
   * IntentB (0.60), IntentC (0.50). The response from the <code>RecognizeText</code>
   * operation would be:</p> <ul> <li> <p>AMAZON.FallbackIntent</p> </li> <li>
   * <p>IntentA</p> </li> <li> <p>IntentB</p> </li> <li> <p>IntentC</p> </li> </ul>
   */
  inline double GetNluIntentConfidenceThreshold() const { return m_nluIntentConfidenceThreshold; }
  inline bool NluIntentConfidenceThresholdHasBeenSet() const { return m_nluIntentConfidenceThresholdHasBeenSet; }
  inline void SetNluIntentConfidenceThreshold(double value) {
    m_nluIntentConfidenceThresholdHasBeenSet = true;
    m_nluIntentConfidenceThreshold = value;
  }
  inline CreateBotLocaleRequest& WithNluIntentConfidenceThreshold(double value) {
    SetNluIntentConfidenceThreshold(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Polly voice ID that Amazon Lex uses for voice interaction with the
   * user.</p>
   */
  inline const VoiceSettings& GetVoiceSettings() const { return m_voiceSettings; }
  inline bool VoiceSettingsHasBeenSet() const { return m_voiceSettingsHasBeenSet; }
  template <typename VoiceSettingsT = VoiceSettings>
  void SetVoiceSettings(VoiceSettingsT&& value) {
    m_voiceSettingsHasBeenSet = true;
    m_voiceSettings = std::forward<VoiceSettingsT>(value);
  }
  template <typename VoiceSettingsT = VoiceSettings>
  CreateBotLocaleRequest& WithVoiceSettings(VoiceSettingsT&& value) {
    SetVoiceSettings(std::forward<VoiceSettingsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Unified speech settings to configure for the new bot locale.</p>
   */
  inline const UnifiedSpeechSettings& GetUnifiedSpeechSettings() const { return m_unifiedSpeechSettings; }
  inline bool UnifiedSpeechSettingsHasBeenSet() const { return m_unifiedSpeechSettingsHasBeenSet; }
  template <typename UnifiedSpeechSettingsT = UnifiedSpeechSettings>
  void SetUnifiedSpeechSettings(UnifiedSpeechSettingsT&& value) {
    m_unifiedSpeechSettingsHasBeenSet = true;
    m_unifiedSpeechSettings = std::forward<UnifiedSpeechSettingsT>(value);
  }
  template <typename UnifiedSpeechSettingsT = UnifiedSpeechSettings>
  CreateBotLocaleRequest& WithUnifiedSpeechSettings(UnifiedSpeechSettingsT&& value) {
    SetUnifiedSpeechSettings(std::forward<UnifiedSpeechSettingsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Speech-to-text settings to configure for the new bot locale.</p>
   */
  inline const SpeechRecognitionSettings& GetSpeechRecognitionSettings() const { return m_speechRecognitionSettings; }
  inline bool SpeechRecognitionSettingsHasBeenSet() const { return m_speechRecognitionSettingsHasBeenSet; }
  template <typename SpeechRecognitionSettingsT = SpeechRecognitionSettings>
  void SetSpeechRecognitionSettings(SpeechRecognitionSettingsT&& value) {
    m_speechRecognitionSettingsHasBeenSet = true;
    m_speechRecognitionSettings = std::forward<SpeechRecognitionSettingsT>(value);
  }
  template <typename SpeechRecognitionSettingsT = SpeechRecognitionSettings>
  CreateBotLocaleRequest& WithSpeechRecognitionSettings(SpeechRecognitionSettingsT&& value) {
    SetSpeechRecognitionSettings(std::forward<SpeechRecognitionSettingsT>(value));
    return *this;
  }
  ///@}

  ///@{

  inline const GenerativeAISettings& GetGenerativeAISettings() const { return m_generativeAISettings; }
  inline bool GenerativeAISettingsHasBeenSet() const { return m_generativeAISettingsHasBeenSet; }
  template <typename GenerativeAISettingsT = GenerativeAISettings>
  void SetGenerativeAISettings(GenerativeAISettingsT&& value) {
    m_generativeAISettingsHasBeenSet = true;
    m_generativeAISettings = std::forward<GenerativeAISettingsT>(value);
  }
  template <typename GenerativeAISettingsT = GenerativeAISettings>
  CreateBotLocaleRequest& WithGenerativeAISettings(GenerativeAISettingsT&& value) {
    SetGenerativeAISettings(std::forward<GenerativeAISettingsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The sensitivity level for voice activity detection (VAD) in the bot locale.
   * This setting helps optimize speech recognition accuracy by adjusting how the
   * system responds to background noise during voice interactions.</p>
   */
  inline SpeechDetectionSensitivity GetSpeechDetectionSensitivity() const { return m_speechDetectionSensitivity; }
  inline bool SpeechDetectionSensitivityHasBeenSet() const { return m_speechDetectionSensitivityHasBeenSet; }
  inline void SetSpeechDetectionSensitivity(SpeechDetectionSensitivity value) {
    m_speechDetectionSensitivityHasBeenSet = true;
    m_speechDetectionSensitivity = value;
  }
  inline CreateBotLocaleRequest& WithSpeechDetectionSensitivity(SpeechDetectionSensitivity value) {
    SetSpeechDetectionSensitivity(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_botId;
  bool m_botIdHasBeenSet = false;

  Aws::String m_botVersion;
  bool m_botVersionHasBeenSet = false;

  Aws::String m_localeId;
  bool m_localeIdHasBeenSet = false;

  Aws::String m_description;
  bool m_descriptionHasBeenSet = false;

  double m_nluIntentConfidenceThreshold{0.0};
  bool m_nluIntentConfidenceThresholdHasBeenSet = false;

  VoiceSettings m_voiceSettings;
  bool m_voiceSettingsHasBeenSet = false;

  UnifiedSpeechSettings m_unifiedSpeechSettings;
  bool m_unifiedSpeechSettingsHasBeenSet = false;

  SpeechRecognitionSettings m_speechRecognitionSettings;
  bool m_speechRecognitionSettingsHasBeenSet = false;

  GenerativeAISettings m_generativeAISettings;
  bool m_generativeAISettingsHasBeenSet = false;

  SpeechDetectionSensitivity m_speechDetectionSensitivity{SpeechDetectionSensitivity::NOT_SET};
  bool m_speechDetectionSensitivityHasBeenSet = false;
};

}  // namespace Model
}  // namespace LexModelsV2
}  // namespace Aws
