﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/DateTime.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/lightsail/LightsailRequest.h>
#include <aws/lightsail/Lightsail_EXPORTS.h>
#include <aws/lightsail/model/LoadBalancerMetricName.h>
#include <aws/lightsail/model/MetricStatistic.h>
#include <aws/lightsail/model/MetricUnit.h>

#include <utility>

namespace Aws {
namespace Lightsail {
namespace Model {

/**
 */
class GetLoadBalancerMetricDataRequest : public LightsailRequest {
 public:
  AWS_LIGHTSAIL_API GetLoadBalancerMetricDataRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "GetLoadBalancerMetricData"; }

  AWS_LIGHTSAIL_API Aws::String SerializePayload() const override;

  AWS_LIGHTSAIL_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The name of the load balancer.</p>
   */
  inline const Aws::String& GetLoadBalancerName() const { return m_loadBalancerName; }
  inline bool LoadBalancerNameHasBeenSet() const { return m_loadBalancerNameHasBeenSet; }
  template <typename LoadBalancerNameT = Aws::String>
  void SetLoadBalancerName(LoadBalancerNameT&& value) {
    m_loadBalancerNameHasBeenSet = true;
    m_loadBalancerName = std::forward<LoadBalancerNameT>(value);
  }
  template <typename LoadBalancerNameT = Aws::String>
  GetLoadBalancerMetricDataRequest& WithLoadBalancerName(LoadBalancerNameT&& value) {
    SetLoadBalancerName(std::forward<LoadBalancerNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The metric for which you want to return information.</p> <p>Valid load
   * balancer metric names are listed below, along with the most useful
   * <code>statistics</code> to include in your request, and the published
   * <code>unit</code> value.</p> <ul> <li> <p> <b>
   * <code>ClientTLSNegotiationErrorCount</code> </b> - The number of TLS connections
   * initiated by the client that did not establish a session with the load balancer
   * due to a TLS error generated by the load balancer. Possible causes include a
   * mismatch of ciphers or protocols.</p> <p> <code>Statistics</code>: The most
   * useful statistic is <code>Sum</code>.</p> <p> <code>Unit</code>: The published
   * unit is <code>Count</code>.</p> </li> <li> <p> <b> <code>HealthyHostCount</code>
   * </b> - The number of target instances that are considered healthy.</p> <p>
   * <code>Statistics</code>: The most useful statistic are <code>Average</code>,
   * <code>Minimum</code>, and <code>Maximum</code>.</p> <p> <code>Unit</code>: The
   * published unit is <code>Count</code>.</p> </li> <li> <p> <b>
   * <code>HTTPCode_Instance_2XX_Count</code> </b> - The number of HTTP 2XX response
   * codes generated by the target instances. This does not include any response
   * codes generated by the load balancer.</p> <p> <code>Statistics</code>: The most
   * useful statistic is <code>Sum</code>. Note that <code>Minimum</code>,
   * <code>Maximum</code>, and <code>Average</code> all return <code>1</code>.</p>
   * <p> <code>Unit</code>: The published unit is <code>Count</code>.</p> </li> <li>
   * <p> <b> <code>HTTPCode_Instance_3XX_Count</code> </b> - The number of HTTP 3XX
   * response codes generated by the target instances. This does not include any
   * response codes generated by the load balancer.</p> <p> <code>Statistics</code>:
   * The most useful statistic is <code>Sum</code>. Note that <code>Minimum</code>,
   * <code>Maximum</code>, and <code>Average</code> all return <code>1</code>.</p>
   * <p> <code>Unit</code>: The published unit is <code>Count</code>.</p> </li> <li>
   * <p> <b> <code>HTTPCode_Instance_4XX_Count</code> </b> - The number of HTTP 4XX
   * response codes generated by the target instances. This does not include any
   * response codes generated by the load balancer.</p> <p> <code>Statistics</code>:
   * The most useful statistic is <code>Sum</code>. Note that <code>Minimum</code>,
   * <code>Maximum</code>, and <code>Average</code> all return <code>1</code>.</p>
   * <p> <code>Unit</code>: The published unit is <code>Count</code>.</p> </li> <li>
   * <p> <b> <code>HTTPCode_Instance_5XX_Count</code> </b> - The number of HTTP 5XX
   * response codes generated by the target instances. This does not include any
   * response codes generated by the load balancer.</p> <p> <code>Statistics</code>:
   * The most useful statistic is <code>Sum</code>. Note that <code>Minimum</code>,
   * <code>Maximum</code>, and <code>Average</code> all return <code>1</code>.</p>
   * <p> <code>Unit</code>: The published unit is <code>Count</code>.</p> </li> <li>
   * <p> <b> <code>HTTPCode_LB_4XX_Count</code> </b> - The number of HTTP 4XX client
   * error codes that originated from the load balancer. Client errors are generated
   * when requests are malformed or incomplete. These requests were not received by
   * the target instance. This count does not include response codes generated by the
   * target instances.</p> <p> <code>Statistics</code>: The most useful statistic is
   * <code>Sum</code>. Note that <code>Minimum</code>, <code>Maximum</code>, and
   * <code>Average</code> all return <code>1</code>.</p> <p> <code>Unit</code>: The
   * published unit is <code>Count</code>.</p> </li> <li> <p> <b>
   * <code>HTTPCode_LB_5XX_Count</code> </b> - The number of HTTP 5XX server error
   * codes that originated from the load balancer. This does not include any response
   * codes generated by the target instance. This metric is reported if there are no
   * healthy instances attached to the load balancer, or if the request rate exceeds
   * the capacity of the instances (spillover) or the load balancer.</p> <p>
   * <code>Statistics</code>: The most useful statistic is <code>Sum</code>. Note
   * that <code>Minimum</code>, <code>Maximum</code>, and <code>Average</code> all
   * return <code>1</code>.</p> <p> <code>Unit</code>: The published unit is
   * <code>Count</code>.</p> </li> <li> <p> <b> <code>InstanceResponseTime</code>
   * </b> - The time elapsed, in seconds, after the request leaves the load balancer
   * until a response from the target instance is received.</p> <p>
   * <code>Statistics</code>: The most useful statistic is <code>Average</code>.</p>
   * <p> <code>Unit</code>: The published unit is <code>Seconds</code>.</p> </li>
   * <li> <p> <b> <code>RejectedConnectionCount</code> </b> - The number of
   * connections that were rejected because the load balancer had reached its maximum
   * number of connections.</p> <p> <code>Statistics</code>: The most useful
   * statistic is <code>Sum</code>.</p> <p> <code>Unit</code>: The published unit is
   * <code>Count</code>.</p> </li> <li> <p> <b> <code>RequestCount</code> </b> - The
   * number of requests processed over IPv4. This count includes only the requests
   * with a response generated by a target instance of the load balancer.</p> <p>
   * <code>Statistics</code>: The most useful statistic is <code>Sum</code>. Note
   * that <code>Minimum</code>, <code>Maximum</code>, and <code>Average</code> all
   * return <code>1</code>.</p> <p> <code>Unit</code>: The published unit is
   * <code>Count</code>.</p> </li> <li> <p> <b> <code>UnhealthyHostCount</code> </b>
   * - The number of target instances that are considered unhealthy.</p> <p>
   * <code>Statistics</code>: The most useful statistic are <code>Average</code>,
   * <code>Minimum</code>, and <code>Maximum</code>.</p> <p> <code>Unit</code>: The
   * published unit is <code>Count</code>.</p> </li> </ul>
   */
  inline LoadBalancerMetricName GetMetricName() const { return m_metricName; }
  inline bool MetricNameHasBeenSet() const { return m_metricNameHasBeenSet; }
  inline void SetMetricName(LoadBalancerMetricName value) {
    m_metricNameHasBeenSet = true;
    m_metricName = value;
  }
  inline GetLoadBalancerMetricDataRequest& WithMetricName(LoadBalancerMetricName value) {
    SetMetricName(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The granularity, in seconds, of the returned data points.</p>
   */
  inline int GetPeriod() const { return m_period; }
  inline bool PeriodHasBeenSet() const { return m_periodHasBeenSet; }
  inline void SetPeriod(int value) {
    m_periodHasBeenSet = true;
    m_period = value;
  }
  inline GetLoadBalancerMetricDataRequest& WithPeriod(int value) {
    SetPeriod(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The start time of the period.</p>
   */
  inline const Aws::Utils::DateTime& GetStartTime() const { return m_startTime; }
  inline bool StartTimeHasBeenSet() const { return m_startTimeHasBeenSet; }
  template <typename StartTimeT = Aws::Utils::DateTime>
  void SetStartTime(StartTimeT&& value) {
    m_startTimeHasBeenSet = true;
    m_startTime = std::forward<StartTimeT>(value);
  }
  template <typename StartTimeT = Aws::Utils::DateTime>
  GetLoadBalancerMetricDataRequest& WithStartTime(StartTimeT&& value) {
    SetStartTime(std::forward<StartTimeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The end time of the period.</p>
   */
  inline const Aws::Utils::DateTime& GetEndTime() const { return m_endTime; }
  inline bool EndTimeHasBeenSet() const { return m_endTimeHasBeenSet; }
  template <typename EndTimeT = Aws::Utils::DateTime>
  void SetEndTime(EndTimeT&& value) {
    m_endTimeHasBeenSet = true;
    m_endTime = std::forward<EndTimeT>(value);
  }
  template <typename EndTimeT = Aws::Utils::DateTime>
  GetLoadBalancerMetricDataRequest& WithEndTime(EndTimeT&& value) {
    SetEndTime(std::forward<EndTimeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The unit for the metric data request. Valid units depend on the metric data
   * being requested. For the valid units with each available metric, see the
   * <code>metricName</code> parameter.</p>
   */
  inline MetricUnit GetUnit() const { return m_unit; }
  inline bool UnitHasBeenSet() const { return m_unitHasBeenSet; }
  inline void SetUnit(MetricUnit value) {
    m_unitHasBeenSet = true;
    m_unit = value;
  }
  inline GetLoadBalancerMetricDataRequest& WithUnit(MetricUnit value) {
    SetUnit(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The statistic for the metric.</p> <p>The following statistics are
   * available:</p> <ul> <li> <p> <code>Minimum</code> - The lowest value observed
   * during the specified period. Use this value to determine low volumes of activity
   * for your application.</p> </li> <li> <p> <code>Maximum</code> - The highest
   * value observed during the specified period. Use this value to determine high
   * volumes of activity for your application.</p> </li> <li> <p> <code>Sum</code> -
   * All values submitted for the matching metric added together. You can use this
   * statistic to determine the total volume of a metric.</p> </li> <li> <p>
   * <code>Average</code> - The value of Sum / SampleCount during the specified
   * period. By comparing this statistic with the Minimum and Maximum values, you can
   * determine the full scope of a metric and how close the average use is to the
   * Minimum and Maximum values. This comparison helps you to know when to increase
   * or decrease your resources.</p> </li> <li> <p> <code>SampleCount</code> - The
   * count, or number, of data points used for the statistical calculation.</p> </li>
   * </ul>
   */
  inline const Aws::Vector<MetricStatistic>& GetStatistics() const { return m_statistics; }
  inline bool StatisticsHasBeenSet() const { return m_statisticsHasBeenSet; }
  template <typename StatisticsT = Aws::Vector<MetricStatistic>>
  void SetStatistics(StatisticsT&& value) {
    m_statisticsHasBeenSet = true;
    m_statistics = std::forward<StatisticsT>(value);
  }
  template <typename StatisticsT = Aws::Vector<MetricStatistic>>
  GetLoadBalancerMetricDataRequest& WithStatistics(StatisticsT&& value) {
    SetStatistics(std::forward<StatisticsT>(value));
    return *this;
  }
  inline GetLoadBalancerMetricDataRequest& AddStatistics(MetricStatistic value) {
    m_statisticsHasBeenSet = true;
    m_statistics.push_back(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_loadBalancerName;
  bool m_loadBalancerNameHasBeenSet = false;

  LoadBalancerMetricName m_metricName{LoadBalancerMetricName::NOT_SET};
  bool m_metricNameHasBeenSet = false;

  int m_period{0};
  bool m_periodHasBeenSet = false;

  Aws::Utils::DateTime m_startTime{};
  bool m_startTimeHasBeenSet = false;

  Aws::Utils::DateTime m_endTime{};
  bool m_endTimeHasBeenSet = false;

  MetricUnit m_unit{MetricUnit::NOT_SET};
  bool m_unitHasBeenSet = false;

  Aws::Vector<MetricStatistic> m_statistics;
  bool m_statisticsHasBeenSet = false;
};

}  // namespace Model
}  // namespace Lightsail
}  // namespace Aws
