﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/UUID.h>
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/medialive/MediaLiveRequest.h>
#include <aws/medialive/MediaLive_EXPORTS.h>
#include <aws/medialive/model/RenewalSettings.h>

#include <utility>

namespace Aws {
namespace MediaLive {
namespace Model {

/**
 * Placeholder documentation for PurchaseOfferingRequest<p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/medialive-2017-10-14/PurchaseOfferingRequest">AWS
 * API Reference</a></p>
 */
class PurchaseOfferingRequest : public MediaLiveRequest {
 public:
  AWS_MEDIALIVE_API PurchaseOfferingRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "PurchaseOffering"; }

  AWS_MEDIALIVE_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * Number of resources
   */
  inline int GetCount() const { return m_count; }
  inline bool CountHasBeenSet() const { return m_countHasBeenSet; }
  inline void SetCount(int value) {
    m_countHasBeenSet = true;
    m_count = value;
  }
  inline PurchaseOfferingRequest& WithCount(int value) {
    SetCount(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * Name for the new reservation
   */
  inline const Aws::String& GetName() const { return m_name; }
  inline bool NameHasBeenSet() const { return m_nameHasBeenSet; }
  template <typename NameT = Aws::String>
  void SetName(NameT&& value) {
    m_nameHasBeenSet = true;
    m_name = std::forward<NameT>(value);
  }
  template <typename NameT = Aws::String>
  PurchaseOfferingRequest& WithName(NameT&& value) {
    SetName(std::forward<NameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * Offering to purchase, e.g. '87654321'
   */
  inline const Aws::String& GetOfferingId() const { return m_offeringId; }
  inline bool OfferingIdHasBeenSet() const { return m_offeringIdHasBeenSet; }
  template <typename OfferingIdT = Aws::String>
  void SetOfferingId(OfferingIdT&& value) {
    m_offeringIdHasBeenSet = true;
    m_offeringId = std::forward<OfferingIdT>(value);
  }
  template <typename OfferingIdT = Aws::String>
  PurchaseOfferingRequest& WithOfferingId(OfferingIdT&& value) {
    SetOfferingId(std::forward<OfferingIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * Renewal settings for the reservation
   */
  inline const RenewalSettings& GetRenewalSettings() const { return m_renewalSettings; }
  inline bool RenewalSettingsHasBeenSet() const { return m_renewalSettingsHasBeenSet; }
  template <typename RenewalSettingsT = RenewalSettings>
  void SetRenewalSettings(RenewalSettingsT&& value) {
    m_renewalSettingsHasBeenSet = true;
    m_renewalSettings = std::forward<RenewalSettingsT>(value);
  }
  template <typename RenewalSettingsT = RenewalSettings>
  PurchaseOfferingRequest& WithRenewalSettings(RenewalSettingsT&& value) {
    SetRenewalSettings(std::forward<RenewalSettingsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * Unique request ID to be specified. This is needed to prevent retries from
   * creating multiple resources.
   */
  inline const Aws::String& GetRequestId() const { return m_requestId; }
  inline bool RequestIdHasBeenSet() const { return m_requestIdHasBeenSet; }
  template <typename RequestIdT = Aws::String>
  void SetRequestId(RequestIdT&& value) {
    m_requestIdHasBeenSet = true;
    m_requestId = std::forward<RequestIdT>(value);
  }
  template <typename RequestIdT = Aws::String>
  PurchaseOfferingRequest& WithRequestId(RequestIdT&& value) {
    SetRequestId(std::forward<RequestIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * Requested reservation start time (UTC) in ISO-8601 format. The specified time
   * must be between the first day of the current month and one year from now. If no
   * value is given, the default is now.
   */
  inline const Aws::String& GetStart() const { return m_start; }
  inline bool StartHasBeenSet() const { return m_startHasBeenSet; }
  template <typename StartT = Aws::String>
  void SetStart(StartT&& value) {
    m_startHasBeenSet = true;
    m_start = std::forward<StartT>(value);
  }
  template <typename StartT = Aws::String>
  PurchaseOfferingRequest& WithStart(StartT&& value) {
    SetStart(std::forward<StartT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * A collection of key-value pairs
   */
  inline const Aws::Map<Aws::String, Aws::String>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  PurchaseOfferingRequest& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsKeyT = Aws::String, typename TagsValueT = Aws::String>
  PurchaseOfferingRequest& AddTags(TagsKeyT&& key, TagsValueT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace(std::forward<TagsKeyT>(key), std::forward<TagsValueT>(value));
    return *this;
  }
  ///@}
 private:
  int m_count{0};
  bool m_countHasBeenSet = false;

  Aws::String m_name;
  bool m_nameHasBeenSet = false;

  Aws::String m_offeringId;
  bool m_offeringIdHasBeenSet = false;

  RenewalSettings m_renewalSettings;
  bool m_renewalSettingsHasBeenSet = false;

  Aws::String m_requestId{Aws::Utils::UUID::PseudoRandomUUID()};
  bool m_requestIdHasBeenSet = true;

  Aws::String m_start;
  bool m_startHasBeenSet = false;

  Aws::Map<Aws::String, Aws::String> m_tags;
  bool m_tagsHasBeenSet = false;
};

}  // namespace Model
}  // namespace MediaLive
}  // namespace Aws
