﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/DateTime.h>
#include <aws/core/utils/memory/stl/AWSStreamFwd.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/monitoring/CloudWatch_EXPORTS.h>
#include <aws/monitoring/model/MessageData.h>
#include <aws/monitoring/model/StatusCode.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Xml {
class XmlNode;
}  // namespace Xml
}  // namespace Utils
namespace CloudWatch {
namespace Model {

/**
 * <p>A <code>GetMetricData</code> call returns an array of
 * <code>MetricDataResult</code> structures. Each of these structures includes the
 * data points for that metric, along with the timestamps of those data points and
 * other identifying information.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/monitoring-2010-08-01/MetricDataResult">AWS
 * API Reference</a></p>
 */
class MetricDataResult {
 public:
  AWS_CLOUDWATCH_API MetricDataResult() = default;
  AWS_CLOUDWATCH_API MetricDataResult(const Aws::Utils::Xml::XmlNode& xmlNode);
  AWS_CLOUDWATCH_API MetricDataResult& operator=(const Aws::Utils::Xml::XmlNode& xmlNode);

  AWS_CLOUDWATCH_API void OutputToStream(Aws::OStream& ostream, const char* location, unsigned index, const char* locationValue) const;
  AWS_CLOUDWATCH_API void OutputToStream(Aws::OStream& oStream, const char* location) const;

  ///@{
  /**
   * <p>The short name you specified to represent this metric.</p>
   */
  inline const Aws::String& GetId() const { return m_id; }
  inline bool IdHasBeenSet() const { return m_idHasBeenSet; }
  template <typename IdT = Aws::String>
  void SetId(IdT&& value) {
    m_idHasBeenSet = true;
    m_id = std::forward<IdT>(value);
  }
  template <typename IdT = Aws::String>
  MetricDataResult& WithId(IdT&& value) {
    SetId(std::forward<IdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The human-readable label associated with the data.</p>
   */
  inline const Aws::String& GetLabel() const { return m_label; }
  inline bool LabelHasBeenSet() const { return m_labelHasBeenSet; }
  template <typename LabelT = Aws::String>
  void SetLabel(LabelT&& value) {
    m_labelHasBeenSet = true;
    m_label = std::forward<LabelT>(value);
  }
  template <typename LabelT = Aws::String>
  MetricDataResult& WithLabel(LabelT&& value) {
    SetLabel(std::forward<LabelT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The timestamps for the data points, formatted in Unix timestamp format. The
   * number of timestamps always matches the number of values and the value for
   * Timestamps[x] is Values[x].</p>
   */
  inline const Aws::Vector<Aws::Utils::DateTime>& GetTimestamps() const { return m_timestamps; }
  inline bool TimestampsHasBeenSet() const { return m_timestampsHasBeenSet; }
  template <typename TimestampsT = Aws::Vector<Aws::Utils::DateTime>>
  void SetTimestamps(TimestampsT&& value) {
    m_timestampsHasBeenSet = true;
    m_timestamps = std::forward<TimestampsT>(value);
  }
  template <typename TimestampsT = Aws::Vector<Aws::Utils::DateTime>>
  MetricDataResult& WithTimestamps(TimestampsT&& value) {
    SetTimestamps(std::forward<TimestampsT>(value));
    return *this;
  }
  template <typename TimestampsT = Aws::Utils::DateTime>
  MetricDataResult& AddTimestamps(TimestampsT&& value) {
    m_timestampsHasBeenSet = true;
    m_timestamps.emplace_back(std::forward<TimestampsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The data points for the metric corresponding to <code>Timestamps</code>. The
   * number of values always matches the number of timestamps and the timestamp for
   * Values[x] is Timestamps[x].</p>
   */
  inline const Aws::Vector<double>& GetValues() const { return m_values; }
  inline bool ValuesHasBeenSet() const { return m_valuesHasBeenSet; }
  template <typename ValuesT = Aws::Vector<double>>
  void SetValues(ValuesT&& value) {
    m_valuesHasBeenSet = true;
    m_values = std::forward<ValuesT>(value);
  }
  template <typename ValuesT = Aws::Vector<double>>
  MetricDataResult& WithValues(ValuesT&& value) {
    SetValues(std::forward<ValuesT>(value));
    return *this;
  }
  inline MetricDataResult& AddValues(double value) {
    m_valuesHasBeenSet = true;
    m_values.push_back(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The status of the returned data. <code>Complete</code> indicates that all
   * data points in the requested time range were returned. <code>PartialData</code>
   * means that an incomplete set of data points were returned. You can use the
   * <code>NextToken</code> value that was returned and repeat your request to get
   * more data points. <code>NextToken</code> is not returned if you are performing a
   * math expression. <code>InternalError</code> indicates that an error occurred.
   * Retry your request using <code>NextToken</code>, if present.</p>
   */
  inline StatusCode GetStatusCode() const { return m_statusCode; }
  inline bool StatusCodeHasBeenSet() const { return m_statusCodeHasBeenSet; }
  inline void SetStatusCode(StatusCode value) {
    m_statusCodeHasBeenSet = true;
    m_statusCode = value;
  }
  inline MetricDataResult& WithStatusCode(StatusCode value) {
    SetStatusCode(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A list of messages with additional information about the data returned.</p>
   */
  inline const Aws::Vector<MessageData>& GetMessages() const { return m_messages; }
  inline bool MessagesHasBeenSet() const { return m_messagesHasBeenSet; }
  template <typename MessagesT = Aws::Vector<MessageData>>
  void SetMessages(MessagesT&& value) {
    m_messagesHasBeenSet = true;
    m_messages = std::forward<MessagesT>(value);
  }
  template <typename MessagesT = Aws::Vector<MessageData>>
  MetricDataResult& WithMessages(MessagesT&& value) {
    SetMessages(std::forward<MessagesT>(value));
    return *this;
  }
  template <typename MessagesT = MessageData>
  MetricDataResult& AddMessages(MessagesT&& value) {
    m_messagesHasBeenSet = true;
    m_messages.emplace_back(std::forward<MessagesT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_id;
  bool m_idHasBeenSet = false;

  Aws::String m_label;
  bool m_labelHasBeenSet = false;

  Aws::Vector<Aws::Utils::DateTime> m_timestamps;
  bool m_timestampsHasBeenSet = false;

  Aws::Vector<double> m_values;
  bool m_valuesHasBeenSet = false;

  StatusCode m_statusCode{StatusCode::NOT_SET};
  bool m_statusCodeHasBeenSet = false;

  Aws::Vector<MessageData> m_messages;
  bool m_messagesHasBeenSet = false;
};

}  // namespace Model
}  // namespace CloudWatch
}  // namespace Aws
