﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSStreamFwd.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/neptune/Neptune_EXPORTS.h>
#include <aws/neptune/model/PendingCloudwatchLogsExports.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Xml {
class XmlNode;
}  // namespace Xml
}  // namespace Utils
namespace Neptune {
namespace Model {

/**
 * <p>This data type is used as a response element in the
 * <code>ModifyDBCluster</code> operation and contains changes that will be applied
 * during the next maintenance window.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/neptune-2014-10-31/ClusterPendingModifiedValues">AWS
 * API Reference</a></p>
 */
class ClusterPendingModifiedValues {
 public:
  AWS_NEPTUNE_API ClusterPendingModifiedValues() = default;
  AWS_NEPTUNE_API ClusterPendingModifiedValues(const Aws::Utils::Xml::XmlNode& xmlNode);
  AWS_NEPTUNE_API ClusterPendingModifiedValues& operator=(const Aws::Utils::Xml::XmlNode& xmlNode);

  AWS_NEPTUNE_API void OutputToStream(Aws::OStream& ostream, const char* location, unsigned index, const char* locationValue) const;
  AWS_NEPTUNE_API void OutputToStream(Aws::OStream& oStream, const char* location) const;

  ///@{
  /**
   * <p>This <code>PendingCloudwatchLogsExports</code> structure specifies pending
   * changes to which CloudWatch logs are enabled and which are disabled.</p>
   */
  inline const PendingCloudwatchLogsExports& GetPendingCloudwatchLogsExports() const { return m_pendingCloudwatchLogsExports; }
  inline bool PendingCloudwatchLogsExportsHasBeenSet() const { return m_pendingCloudwatchLogsExportsHasBeenSet; }
  template <typename PendingCloudwatchLogsExportsT = PendingCloudwatchLogsExports>
  void SetPendingCloudwatchLogsExports(PendingCloudwatchLogsExportsT&& value) {
    m_pendingCloudwatchLogsExportsHasBeenSet = true;
    m_pendingCloudwatchLogsExports = std::forward<PendingCloudwatchLogsExportsT>(value);
  }
  template <typename PendingCloudwatchLogsExportsT = PendingCloudwatchLogsExports>
  ClusterPendingModifiedValues& WithPendingCloudwatchLogsExports(PendingCloudwatchLogsExportsT&& value) {
    SetPendingCloudwatchLogsExports(std::forward<PendingCloudwatchLogsExportsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The DBClusterIdentifier value for the DB cluster.</p>
   */
  inline const Aws::String& GetDBClusterIdentifier() const { return m_dBClusterIdentifier; }
  inline bool DBClusterIdentifierHasBeenSet() const { return m_dBClusterIdentifierHasBeenSet; }
  template <typename DBClusterIdentifierT = Aws::String>
  void SetDBClusterIdentifier(DBClusterIdentifierT&& value) {
    m_dBClusterIdentifierHasBeenSet = true;
    m_dBClusterIdentifier = std::forward<DBClusterIdentifierT>(value);
  }
  template <typename DBClusterIdentifierT = Aws::String>
  ClusterPendingModifiedValues& WithDBClusterIdentifier(DBClusterIdentifierT&& value) {
    SetDBClusterIdentifier(std::forward<DBClusterIdentifierT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A value that indicates whether mapping of Amazon Web Services Identity and
   * Access Management (IAM) accounts to database accounts is enabled.</p>
   */
  inline bool GetIAMDatabaseAuthenticationEnabled() const { return m_iAMDatabaseAuthenticationEnabled; }
  inline bool IAMDatabaseAuthenticationEnabledHasBeenSet() const { return m_iAMDatabaseAuthenticationEnabledHasBeenSet; }
  inline void SetIAMDatabaseAuthenticationEnabled(bool value) {
    m_iAMDatabaseAuthenticationEnabledHasBeenSet = true;
    m_iAMDatabaseAuthenticationEnabled = value;
  }
  inline ClusterPendingModifiedValues& WithIAMDatabaseAuthenticationEnabled(bool value) {
    SetIAMDatabaseAuthenticationEnabled(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The database engine version.</p>
   */
  inline const Aws::String& GetEngineVersion() const { return m_engineVersion; }
  inline bool EngineVersionHasBeenSet() const { return m_engineVersionHasBeenSet; }
  template <typename EngineVersionT = Aws::String>
  void SetEngineVersion(EngineVersionT&& value) {
    m_engineVersionHasBeenSet = true;
    m_engineVersion = std::forward<EngineVersionT>(value);
  }
  template <typename EngineVersionT = Aws::String>
  ClusterPendingModifiedValues& WithEngineVersion(EngineVersionT&& value) {
    SetEngineVersion(std::forward<EngineVersionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The number of days for which automatic DB snapshots are retained.</p>
   */
  inline int GetBackupRetentionPeriod() const { return m_backupRetentionPeriod; }
  inline bool BackupRetentionPeriodHasBeenSet() const { return m_backupRetentionPeriodHasBeenSet; }
  inline void SetBackupRetentionPeriod(int value) {
    m_backupRetentionPeriodHasBeenSet = true;
    m_backupRetentionPeriod = value;
  }
  inline ClusterPendingModifiedValues& WithBackupRetentionPeriod(int value) {
    SetBackupRetentionPeriod(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The pending change in storage type for the DB cluster.   Valid Values:</p>
   * <ul> <li> <p> <b> <code>standard</code> </b>   –   ( <i>the default</i> )
   * Configures cost-effective database storage for applications with moderate to
   * small I/O usage.</p> </li> <li> <p> <b> <code>iopt1</code> </b>   –   Enables <a
   * href="https://docs.aws.amazon.com/neptune/latest/userguide/storage-types.html#provisioned-iops-storage">I/O-Optimized
   * storage</a> that's designed to meet the needs of I/O-intensive graph workloads
   * that require predictable pricing with low I/O latency and consistent I/O
   * throughput.</p> <p>Neptune I/O-Optimized storage is only available starting with
   * engine release 1.3.0.0.</p> </li> </ul>
   */
  inline const Aws::String& GetStorageType() const { return m_storageType; }
  inline bool StorageTypeHasBeenSet() const { return m_storageTypeHasBeenSet; }
  template <typename StorageTypeT = Aws::String>
  void SetStorageType(StorageTypeT&& value) {
    m_storageTypeHasBeenSet = true;
    m_storageType = std::forward<StorageTypeT>(value);
  }
  template <typename StorageTypeT = Aws::String>
  ClusterPendingModifiedValues& WithStorageType(StorageTypeT&& value) {
    SetStorageType(std::forward<StorageTypeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The allocated storage size in gibibytes (GiB) for database engines. For
   * Neptune, <code>AllocatedStorage</code> always returns 1, because Neptune DB
   * cluster storage size isn't fixed, but instead automatically adjusts as
   * needed.</p>
   */
  inline int GetAllocatedStorage() const { return m_allocatedStorage; }
  inline bool AllocatedStorageHasBeenSet() const { return m_allocatedStorageHasBeenSet; }
  inline void SetAllocatedStorage(int value) {
    m_allocatedStorageHasBeenSet = true;
    m_allocatedStorage = value;
  }
  inline ClusterPendingModifiedValues& WithAllocatedStorage(int value) {
    SetAllocatedStorage(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Provisioned IOPS (I/O operations per second) value. This setting is only
   * for Multi-AZ DB clusters.</p>
   */
  inline int GetIops() const { return m_iops; }
  inline bool IopsHasBeenSet() const { return m_iopsHasBeenSet; }
  inline void SetIops(int value) {
    m_iopsHasBeenSet = true;
    m_iops = value;
  }
  inline ClusterPendingModifiedValues& WithIops(int value) {
    SetIops(value);
    return *this;
  }
  ///@}
 private:
  PendingCloudwatchLogsExports m_pendingCloudwatchLogsExports;
  bool m_pendingCloudwatchLogsExportsHasBeenSet = false;

  Aws::String m_dBClusterIdentifier;
  bool m_dBClusterIdentifierHasBeenSet = false;

  bool m_iAMDatabaseAuthenticationEnabled{false};
  bool m_iAMDatabaseAuthenticationEnabledHasBeenSet = false;

  Aws::String m_engineVersion;
  bool m_engineVersionHasBeenSet = false;

  int m_backupRetentionPeriod{0};
  bool m_backupRetentionPeriodHasBeenSet = false;

  Aws::String m_storageType;
  bool m_storageTypeHasBeenSet = false;

  int m_allocatedStorage{0};
  bool m_allocatedStorageHasBeenSet = false;

  int m_iops{0};
  bool m_iopsHasBeenSet = false;
};

}  // namespace Model
}  // namespace Neptune
}  // namespace Aws
