﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/notifications/NotificationsRequest.h>
#include <aws/notifications/Notifications_EXPORTS.h>

#include <utility>

namespace Aws {
namespace Notifications {
namespace Model {

/**
 */
class CreateEventRuleRequest : public NotificationsRequest {
 public:
  AWS_NOTIFICATIONS_API CreateEventRuleRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateEventRule"; }

  AWS_NOTIFICATIONS_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the <code>NotificationConfiguration</code>
   * associated with this <code>EventRule</code>.</p>
   */
  inline const Aws::String& GetNotificationConfigurationArn() const { return m_notificationConfigurationArn; }
  inline bool NotificationConfigurationArnHasBeenSet() const { return m_notificationConfigurationArnHasBeenSet; }
  template <typename NotificationConfigurationArnT = Aws::String>
  void SetNotificationConfigurationArn(NotificationConfigurationArnT&& value) {
    m_notificationConfigurationArnHasBeenSet = true;
    m_notificationConfigurationArn = std::forward<NotificationConfigurationArnT>(value);
  }
  template <typename NotificationConfigurationArnT = Aws::String>
  CreateEventRuleRequest& WithNotificationConfigurationArn(NotificationConfigurationArnT&& value) {
    SetNotificationConfigurationArn(std::forward<NotificationConfigurationArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The matched event source.</p> <p>Must match one of the valid EventBridge
   * sources. Only Amazon Web Services service sourced events are supported. For
   * example, <code>aws.ec2</code> and <code>aws.cloudwatch</code>. For more
   * information, see <a
   * href="https://docs.aws.amazon.com/eventbridge/latest/userguide/eb-service-event.html#eb-service-event-delivery-level">Event
   * delivery from Amazon Web Services services</a> in the <i>Amazon EventBridge User
   * Guide</i>.</p>
   */
  inline const Aws::String& GetSource() const { return m_source; }
  inline bool SourceHasBeenSet() const { return m_sourceHasBeenSet; }
  template <typename SourceT = Aws::String>
  void SetSource(SourceT&& value) {
    m_sourceHasBeenSet = true;
    m_source = std::forward<SourceT>(value);
  }
  template <typename SourceT = Aws::String>
  CreateEventRuleRequest& WithSource(SourceT&& value) {
    SetSource(std::forward<SourceT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The event type to match.</p> <p>Must match one of the valid Amazon
   * EventBridge event types. For example, EC2 Instance State-change Notification and
   * Amazon CloudWatch Alarm State Change. For more information, see <a
   * href="https://docs.aws.amazon.com/eventbridge/latest/userguide/eb-service-event.html#eb-service-event-delivery-level">Event
   * delivery from Amazon Web Services services</a> in the <i>Amazon EventBridge User
   * Guide</i>.</p>
   */
  inline const Aws::String& GetEventType() const { return m_eventType; }
  inline bool EventTypeHasBeenSet() const { return m_eventTypeHasBeenSet; }
  template <typename EventTypeT = Aws::String>
  void SetEventType(EventTypeT&& value) {
    m_eventTypeHasBeenSet = true;
    m_eventType = std::forward<EventTypeT>(value);
  }
  template <typename EventTypeT = Aws::String>
  CreateEventRuleRequest& WithEventType(EventTypeT&& value) {
    SetEventType(std::forward<EventTypeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>An additional event pattern used to further filter the events this
   * <code>EventRule</code> receives.</p> <p>For more information, see <a
   * href="https://docs.aws.amazon.com/eventbridge/latest/userguide/eb-event-patterns.html">Amazon
   * EventBridge event patterns</a> in the <i>Amazon EventBridge User Guide.</i> </p>
   */
  inline const Aws::String& GetEventPattern() const { return m_eventPattern; }
  inline bool EventPatternHasBeenSet() const { return m_eventPatternHasBeenSet; }
  template <typename EventPatternT = Aws::String>
  void SetEventPattern(EventPatternT&& value) {
    m_eventPatternHasBeenSet = true;
    m_eventPattern = std::forward<EventPatternT>(value);
  }
  template <typename EventPatternT = Aws::String>
  CreateEventRuleRequest& WithEventPattern(EventPatternT&& value) {
    SetEventPattern(std::forward<EventPatternT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A list of Amazon Web Services Regions that send events to this
   * <code>EventRule</code>.</p>
   */
  inline const Aws::Vector<Aws::String>& GetRegions() const { return m_regions; }
  inline bool RegionsHasBeenSet() const { return m_regionsHasBeenSet; }
  template <typename RegionsT = Aws::Vector<Aws::String>>
  void SetRegions(RegionsT&& value) {
    m_regionsHasBeenSet = true;
    m_regions = std::forward<RegionsT>(value);
  }
  template <typename RegionsT = Aws::Vector<Aws::String>>
  CreateEventRuleRequest& WithRegions(RegionsT&& value) {
    SetRegions(std::forward<RegionsT>(value));
    return *this;
  }
  template <typename RegionsT = Aws::String>
  CreateEventRuleRequest& AddRegions(RegionsT&& value) {
    m_regionsHasBeenSet = true;
    m_regions.emplace_back(std::forward<RegionsT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_notificationConfigurationArn;
  bool m_notificationConfigurationArnHasBeenSet = false;

  Aws::String m_source;
  bool m_sourceHasBeenSet = false;

  Aws::String m_eventType;
  bool m_eventTypeHasBeenSet = false;

  Aws::String m_eventPattern;
  bool m_eventPatternHasBeenSet = false;

  Aws::Vector<Aws::String> m_regions;
  bool m_regionsHasBeenSet = false;
};

}  // namespace Model
}  // namespace Notifications
}  // namespace Aws
