﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/omics/OmicsRequest.h>
#include <aws/omics/Omics_EXPORTS.h>
#include <aws/omics/model/ReferenceItem.h>
#include <aws/omics/model/SseConfig.h>

#include <utility>

namespace Aws {
namespace Omics {
namespace Model {

/**
 */
class CreateVariantStoreRequest : public OmicsRequest {
 public:
  AWS_OMICS_API CreateVariantStoreRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateVariantStore"; }

  AWS_OMICS_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The genome reference for the store's variants.</p>
   */
  inline const ReferenceItem& GetReference() const { return m_reference; }
  inline bool ReferenceHasBeenSet() const { return m_referenceHasBeenSet; }
  template <typename ReferenceT = ReferenceItem>
  void SetReference(ReferenceT&& value) {
    m_referenceHasBeenSet = true;
    m_reference = std::forward<ReferenceT>(value);
  }
  template <typename ReferenceT = ReferenceItem>
  CreateVariantStoreRequest& WithReference(ReferenceT&& value) {
    SetReference(std::forward<ReferenceT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A name for the store.</p>
   */
  inline const Aws::String& GetName() const { return m_name; }
  inline bool NameHasBeenSet() const { return m_nameHasBeenSet; }
  template <typename NameT = Aws::String>
  void SetName(NameT&& value) {
    m_nameHasBeenSet = true;
    m_name = std::forward<NameT>(value);
  }
  template <typename NameT = Aws::String>
  CreateVariantStoreRequest& WithName(NameT&& value) {
    SetName(std::forward<NameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A description for the store.</p>
   */
  inline const Aws::String& GetDescription() const { return m_description; }
  inline bool DescriptionHasBeenSet() const { return m_descriptionHasBeenSet; }
  template <typename DescriptionT = Aws::String>
  void SetDescription(DescriptionT&& value) {
    m_descriptionHasBeenSet = true;
    m_description = std::forward<DescriptionT>(value);
  }
  template <typename DescriptionT = Aws::String>
  CreateVariantStoreRequest& WithDescription(DescriptionT&& value) {
    SetDescription(std::forward<DescriptionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Tags for the store.</p>
   */
  inline const Aws::Map<Aws::String, Aws::String>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  CreateVariantStoreRequest& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsKeyT = Aws::String, typename TagsValueT = Aws::String>
  CreateVariantStoreRequest& AddTags(TagsKeyT&& key, TagsValueT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace(std::forward<TagsKeyT>(key), std::forward<TagsValueT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Server-side encryption (SSE) settings for the store.</p>
   */
  inline const SseConfig& GetSseConfig() const { return m_sseConfig; }
  inline bool SseConfigHasBeenSet() const { return m_sseConfigHasBeenSet; }
  template <typename SseConfigT = SseConfig>
  void SetSseConfig(SseConfigT&& value) {
    m_sseConfigHasBeenSet = true;
    m_sseConfig = std::forward<SseConfigT>(value);
  }
  template <typename SseConfigT = SseConfig>
  CreateVariantStoreRequest& WithSseConfig(SseConfigT&& value) {
    SetSseConfig(std::forward<SseConfigT>(value));
    return *this;
  }
  ///@}
 private:
  ReferenceItem m_reference;
  bool m_referenceHasBeenSet = false;

  Aws::String m_name;
  bool m_nameHasBeenSet = false;

  Aws::String m_description;
  bool m_descriptionHasBeenSet = false;

  Aws::Map<Aws::String, Aws::String> m_tags;
  bool m_tagsHasBeenSet = false;

  SseConfig m_sseConfig;
  bool m_sseConfigHasBeenSet = false;
};

}  // namespace Model
}  // namespace Omics
}  // namespace Aws
