﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/DateTime.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/pca-connector-ad/PcaConnectorAd_EXPORTS.h>
#include <aws/pca-connector-ad/model/TemplateDefinition.h>
#include <aws/pca-connector-ad/model/TemplateRevision.h>
#include <aws/pca-connector-ad/model/TemplateStatus.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Json {
class JsonValue;
class JsonView;
}  // namespace Json
}  // namespace Utils
namespace PcaConnectorAd {
namespace Model {

/**
 * <p>An Active Directory compatible certificate template. Connectors issue
 * certificates against these templates based on the requestor's Active Directory
 * group membership. </p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/pca-connector-ad-2018-05-10/Template">AWS
 * API Reference</a></p>
 */
class Template {
 public:
  AWS_PCACONNECTORAD_API Template() = default;
  AWS_PCACONNECTORAD_API Template(Aws::Utils::Json::JsonView jsonValue);
  AWS_PCACONNECTORAD_API Template& operator=(Aws::Utils::Json::JsonView jsonValue);
  AWS_PCACONNECTORAD_API Aws::Utils::Json::JsonValue Jsonize() const;

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) that was returned when you called <a
   * href="https://docs.aws.amazon.com/pca-connector-ad/latest/APIReference/API_CreateTemplate.html">CreateTemplate</a>.</p>
   */
  inline const Aws::String& GetArn() const { return m_arn; }
  inline bool ArnHasBeenSet() const { return m_arnHasBeenSet; }
  template <typename ArnT = Aws::String>
  void SetArn(ArnT&& value) {
    m_arnHasBeenSet = true;
    m_arn = std::forward<ArnT>(value);
  }
  template <typename ArnT = Aws::String>
  Template& WithArn(ArnT&& value) {
    SetArn(std::forward<ArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> The Amazon Resource Name (ARN) that was returned when you called <a
   * href="https://docs.aws.amazon.com/pca-connector-ad/latest/APIReference/API_CreateConnector.html">CreateConnector</a>.</p>
   */
  inline const Aws::String& GetConnectorArn() const { return m_connectorArn; }
  inline bool ConnectorArnHasBeenSet() const { return m_connectorArnHasBeenSet; }
  template <typename ConnectorArnT = Aws::String>
  void SetConnectorArn(ConnectorArnT&& value) {
    m_connectorArnHasBeenSet = true;
    m_connectorArn = std::forward<ConnectorArnT>(value);
  }
  template <typename ConnectorArnT = Aws::String>
  Template& WithConnectorArn(ConnectorArnT&& value) {
    SetConnectorArn(std::forward<ConnectorArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The date and time that the template was created.</p>
   */
  inline const Aws::Utils::DateTime& GetCreatedAt() const { return m_createdAt; }
  inline bool CreatedAtHasBeenSet() const { return m_createdAtHasBeenSet; }
  template <typename CreatedAtT = Aws::Utils::DateTime>
  void SetCreatedAt(CreatedAtT&& value) {
    m_createdAtHasBeenSet = true;
    m_createdAt = std::forward<CreatedAtT>(value);
  }
  template <typename CreatedAtT = Aws::Utils::DateTime>
  Template& WithCreatedAt(CreatedAtT&& value) {
    SetCreatedAt(std::forward<CreatedAtT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Template configuration to define the information included in certificates.
   * Define certificate validity and renewal periods, certificate request handling
   * and enrollment options, key usage extensions, application policies, and
   * cryptography settings.</p>
   */
  inline const TemplateDefinition& GetDefinition() const { return m_definition; }
  inline bool DefinitionHasBeenSet() const { return m_definitionHasBeenSet; }
  template <typename DefinitionT = TemplateDefinition>
  void SetDefinition(DefinitionT&& value) {
    m_definitionHasBeenSet = true;
    m_definition = std::forward<DefinitionT>(value);
  }
  template <typename DefinitionT = TemplateDefinition>
  Template& WithDefinition(DefinitionT&& value) {
    SetDefinition(std::forward<DefinitionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Name of the templates. Template names must be unique.</p>
   */
  inline const Aws::String& GetName() const { return m_name; }
  inline bool NameHasBeenSet() const { return m_nameHasBeenSet; }
  template <typename NameT = Aws::String>
  void SetName(NameT&& value) {
    m_nameHasBeenSet = true;
    m_name = std::forward<NameT>(value);
  }
  template <typename NameT = Aws::String>
  Template& WithName(NameT&& value) {
    SetName(std::forward<NameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Object identifier of a template.</p>
   */
  inline const Aws::String& GetObjectIdentifier() const { return m_objectIdentifier; }
  inline bool ObjectIdentifierHasBeenSet() const { return m_objectIdentifierHasBeenSet; }
  template <typename ObjectIdentifierT = Aws::String>
  void SetObjectIdentifier(ObjectIdentifierT&& value) {
    m_objectIdentifierHasBeenSet = true;
    m_objectIdentifier = std::forward<ObjectIdentifierT>(value);
  }
  template <typename ObjectIdentifierT = Aws::String>
  Template& WithObjectIdentifier(ObjectIdentifierT&& value) {
    SetObjectIdentifier(std::forward<ObjectIdentifierT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The template schema version. Template schema versions can be v2, v3, or v4.
   * The template configuration options change based on the template schema
   * version.</p>
   */
  inline int GetPolicySchema() const { return m_policySchema; }
  inline bool PolicySchemaHasBeenSet() const { return m_policySchemaHasBeenSet; }
  inline void SetPolicySchema(int value) {
    m_policySchemaHasBeenSet = true;
    m_policySchema = value;
  }
  inline Template& WithPolicySchema(int value) {
    SetPolicySchema(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The version of the template. Template updates will increment the minor
   * revision. Re-enrolling all certificate holders will increment the major
   * revision.</p>
   */
  inline const TemplateRevision& GetRevision() const { return m_revision; }
  inline bool RevisionHasBeenSet() const { return m_revisionHasBeenSet; }
  template <typename RevisionT = TemplateRevision>
  void SetRevision(RevisionT&& value) {
    m_revisionHasBeenSet = true;
    m_revision = std::forward<RevisionT>(value);
  }
  template <typename RevisionT = TemplateRevision>
  Template& WithRevision(RevisionT&& value) {
    SetRevision(std::forward<RevisionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Status of the template. Status can be creating, active, deleting, or
   * failed.</p>
   */
  inline TemplateStatus GetStatus() const { return m_status; }
  inline bool StatusHasBeenSet() const { return m_statusHasBeenSet; }
  inline void SetStatus(TemplateStatus value) {
    m_statusHasBeenSet = true;
    m_status = value;
  }
  inline Template& WithStatus(TemplateStatus value) {
    SetStatus(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The date and time that the template was updated.</p>
   */
  inline const Aws::Utils::DateTime& GetUpdatedAt() const { return m_updatedAt; }
  inline bool UpdatedAtHasBeenSet() const { return m_updatedAtHasBeenSet; }
  template <typename UpdatedAtT = Aws::Utils::DateTime>
  void SetUpdatedAt(UpdatedAtT&& value) {
    m_updatedAtHasBeenSet = true;
    m_updatedAt = std::forward<UpdatedAtT>(value);
  }
  template <typename UpdatedAtT = Aws::Utils::DateTime>
  Template& WithUpdatedAt(UpdatedAtT&& value) {
    SetUpdatedAt(std::forward<UpdatedAtT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_arn;
  bool m_arnHasBeenSet = false;

  Aws::String m_connectorArn;
  bool m_connectorArnHasBeenSet = false;

  Aws::Utils::DateTime m_createdAt{};
  bool m_createdAtHasBeenSet = false;

  TemplateDefinition m_definition;
  bool m_definitionHasBeenSet = false;

  Aws::String m_name;
  bool m_nameHasBeenSet = false;

  Aws::String m_objectIdentifier;
  bool m_objectIdentifierHasBeenSet = false;

  int m_policySchema{0};
  bool m_policySchemaHasBeenSet = false;

  TemplateRevision m_revision;
  bool m_revisionHasBeenSet = false;

  TemplateStatus m_status{TemplateStatus::NOT_SET};
  bool m_statusHasBeenSet = false;

  Aws::Utils::DateTime m_updatedAt{};
  bool m_updatedAtHasBeenSet = false;
};

}  // namespace Model
}  // namespace PcaConnectorAd
}  // namespace Aws
