﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/DateTime.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/rds/RDSRequest.h>
#include <aws/rds/RDS_EXPORTS.h>
#include <aws/rds/model/Filter.h>
#include <aws/rds/model/SourceType.h>

#include <utility>

namespace Aws {
namespace RDS {
namespace Model {

/**
 * <p/><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/rds-2014-10-31/DescribeEventsMessage">AWS
 * API Reference</a></p>
 */
class DescribeEventsRequest : public RDSRequest {
 public:
  AWS_RDS_API DescribeEventsRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "DescribeEvents"; }

  AWS_RDS_API Aws::String SerializePayload() const override;

 protected:
  AWS_RDS_API void DumpBodyToUrl(Aws::Http::URI& uri) const override;

 public:
  ///@{
  /**
   * <p>The identifier of the event source for which events are returned. If not
   * specified, then all sources are included in the response.</p>
   * <p>Constraints:</p> <ul> <li> <p>If <code>SourceIdentifier</code> is supplied,
   * <code>SourceType</code> must also be provided.</p> </li> <li> <p>If the source
   * type is a DB instance, a <code>DBInstanceIdentifier</code> value must be
   * supplied.</p> </li> <li> <p>If the source type is a DB cluster, a
   * <code>DBClusterIdentifier</code> value must be supplied.</p> </li> <li> <p>If
   * the source type is a DB parameter group, a <code>DBParameterGroupName</code>
   * value must be supplied.</p> </li> <li> <p>If the source type is a DB security
   * group, a <code>DBSecurityGroupName</code> value must be supplied.</p> </li> <li>
   * <p>If the source type is a DB snapshot, a <code>DBSnapshotIdentifier</code>
   * value must be supplied.</p> </li> <li> <p>If the source type is a DB cluster
   * snapshot, a <code>DBClusterSnapshotIdentifier</code> value must be supplied.</p>
   * </li> <li> <p>If the source type is an RDS Proxy, a <code>DBProxyName</code>
   * value must be supplied.</p> </li> <li> <p>Can't end with a hyphen or contain two
   * consecutive hyphens.</p> </li> </ul>
   */
  inline const Aws::String& GetSourceIdentifier() const { return m_sourceIdentifier; }
  inline bool SourceIdentifierHasBeenSet() const { return m_sourceIdentifierHasBeenSet; }
  template <typename SourceIdentifierT = Aws::String>
  void SetSourceIdentifier(SourceIdentifierT&& value) {
    m_sourceIdentifierHasBeenSet = true;
    m_sourceIdentifier = std::forward<SourceIdentifierT>(value);
  }
  template <typename SourceIdentifierT = Aws::String>
  DescribeEventsRequest& WithSourceIdentifier(SourceIdentifierT&& value) {
    SetSourceIdentifier(std::forward<SourceIdentifierT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The event source to retrieve events for. If no value is specified, all events
   * are returned.</p>
   */
  inline SourceType GetSourceType() const { return m_sourceType; }
  inline bool SourceTypeHasBeenSet() const { return m_sourceTypeHasBeenSet; }
  inline void SetSourceType(SourceType value) {
    m_sourceTypeHasBeenSet = true;
    m_sourceType = value;
  }
  inline DescribeEventsRequest& WithSourceType(SourceType value) {
    SetSourceType(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The beginning of the time interval to retrieve events for, specified in ISO
   * 8601 format. For more information about ISO 8601, go to the <a
   * href="http://en.wikipedia.org/wiki/ISO_8601">ISO8601 Wikipedia page.</a> </p>
   * <p>Example: 2009-07-08T18:00Z</p>
   */
  inline const Aws::Utils::DateTime& GetStartTime() const { return m_startTime; }
  inline bool StartTimeHasBeenSet() const { return m_startTimeHasBeenSet; }
  template <typename StartTimeT = Aws::Utils::DateTime>
  void SetStartTime(StartTimeT&& value) {
    m_startTimeHasBeenSet = true;
    m_startTime = std::forward<StartTimeT>(value);
  }
  template <typename StartTimeT = Aws::Utils::DateTime>
  DescribeEventsRequest& WithStartTime(StartTimeT&& value) {
    SetStartTime(std::forward<StartTimeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The end of the time interval for which to retrieve events, specified in ISO
   * 8601 format. For more information about ISO 8601, go to the <a
   * href="http://en.wikipedia.org/wiki/ISO_8601">ISO8601 Wikipedia page.</a> </p>
   * <p>Example: 2009-07-08T18:00Z</p>
   */
  inline const Aws::Utils::DateTime& GetEndTime() const { return m_endTime; }
  inline bool EndTimeHasBeenSet() const { return m_endTimeHasBeenSet; }
  template <typename EndTimeT = Aws::Utils::DateTime>
  void SetEndTime(EndTimeT&& value) {
    m_endTimeHasBeenSet = true;
    m_endTime = std::forward<EndTimeT>(value);
  }
  template <typename EndTimeT = Aws::Utils::DateTime>
  DescribeEventsRequest& WithEndTime(EndTimeT&& value) {
    SetEndTime(std::forward<EndTimeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The number of minutes to retrieve events for.</p> <p>Default: 60</p>
   */
  inline int GetDuration() const { return m_duration; }
  inline bool DurationHasBeenSet() const { return m_durationHasBeenSet; }
  inline void SetDuration(int value) {
    m_durationHasBeenSet = true;
    m_duration = value;
  }
  inline DescribeEventsRequest& WithDuration(int value) {
    SetDuration(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A list of event categories that trigger notifications for a event
   * notification subscription.</p>
   */
  inline const Aws::Vector<Aws::String>& GetEventCategories() const { return m_eventCategories; }
  inline bool EventCategoriesHasBeenSet() const { return m_eventCategoriesHasBeenSet; }
  template <typename EventCategoriesT = Aws::Vector<Aws::String>>
  void SetEventCategories(EventCategoriesT&& value) {
    m_eventCategoriesHasBeenSet = true;
    m_eventCategories = std::forward<EventCategoriesT>(value);
  }
  template <typename EventCategoriesT = Aws::Vector<Aws::String>>
  DescribeEventsRequest& WithEventCategories(EventCategoriesT&& value) {
    SetEventCategories(std::forward<EventCategoriesT>(value));
    return *this;
  }
  template <typename EventCategoriesT = Aws::String>
  DescribeEventsRequest& AddEventCategories(EventCategoriesT&& value) {
    m_eventCategoriesHasBeenSet = true;
    m_eventCategories.emplace_back(std::forward<EventCategoriesT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>This parameter isn't currently supported.</p>
   */
  inline const Aws::Vector<Filter>& GetFilters() const { return m_filters; }
  inline bool FiltersHasBeenSet() const { return m_filtersHasBeenSet; }
  template <typename FiltersT = Aws::Vector<Filter>>
  void SetFilters(FiltersT&& value) {
    m_filtersHasBeenSet = true;
    m_filters = std::forward<FiltersT>(value);
  }
  template <typename FiltersT = Aws::Vector<Filter>>
  DescribeEventsRequest& WithFilters(FiltersT&& value) {
    SetFilters(std::forward<FiltersT>(value));
    return *this;
  }
  template <typename FiltersT = Filter>
  DescribeEventsRequest& AddFilters(FiltersT&& value) {
    m_filtersHasBeenSet = true;
    m_filters.emplace_back(std::forward<FiltersT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The maximum number of records to include in the response. If more records
   * exist than the specified <code>MaxRecords</code> value, a pagination token
   * called a marker is included in the response so that you can retrieve the
   * remaining results.</p> <p>Default: 100</p> <p>Constraints: Minimum 20, maximum
   * 100.</p>
   */
  inline int GetMaxRecords() const { return m_maxRecords; }
  inline bool MaxRecordsHasBeenSet() const { return m_maxRecordsHasBeenSet; }
  inline void SetMaxRecords(int value) {
    m_maxRecordsHasBeenSet = true;
    m_maxRecords = value;
  }
  inline DescribeEventsRequest& WithMaxRecords(int value) {
    SetMaxRecords(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>An optional pagination token provided by a previous DescribeEvents request.
   * If this parameter is specified, the response includes only records beyond the
   * marker, up to the value specified by <code>MaxRecords</code>.</p>
   */
  inline const Aws::String& GetMarker() const { return m_marker; }
  inline bool MarkerHasBeenSet() const { return m_markerHasBeenSet; }
  template <typename MarkerT = Aws::String>
  void SetMarker(MarkerT&& value) {
    m_markerHasBeenSet = true;
    m_marker = std::forward<MarkerT>(value);
  }
  template <typename MarkerT = Aws::String>
  DescribeEventsRequest& WithMarker(MarkerT&& value) {
    SetMarker(std::forward<MarkerT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_sourceIdentifier;
  bool m_sourceIdentifierHasBeenSet = false;

  SourceType m_sourceType{SourceType::NOT_SET};
  bool m_sourceTypeHasBeenSet = false;

  Aws::Utils::DateTime m_startTime{};
  bool m_startTimeHasBeenSet = false;

  Aws::Utils::DateTime m_endTime{};
  bool m_endTimeHasBeenSet = false;

  int m_duration{0};
  bool m_durationHasBeenSet = false;

  Aws::Vector<Aws::String> m_eventCategories;
  bool m_eventCategoriesHasBeenSet = false;

  Aws::Vector<Filter> m_filters;
  bool m_filtersHasBeenSet = false;

  int m_maxRecords{0};
  bool m_maxRecordsHasBeenSet = false;

  Aws::String m_marker;
  bool m_markerHasBeenSet = false;
};

}  // namespace Model
}  // namespace RDS
}  // namespace Aws
