﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSStreamFwd.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/redshift/Redshift_EXPORTS.h>
#include <aws/redshift/model/ParameterApplyType.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Xml {
class XmlNode;
}  // namespace Xml
}  // namespace Utils
namespace Redshift {
namespace Model {

/**
 * <p>Describes a parameter in a cluster parameter group.</p><p><h3>See Also:</h3>
 * <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/redshift-2012-12-01/Parameter">AWS
 * API Reference</a></p>
 */
class Parameter {
 public:
  AWS_REDSHIFT_API Parameter() = default;
  AWS_REDSHIFT_API Parameter(const Aws::Utils::Xml::XmlNode& xmlNode);
  AWS_REDSHIFT_API Parameter& operator=(const Aws::Utils::Xml::XmlNode& xmlNode);

  AWS_REDSHIFT_API void OutputToStream(Aws::OStream& ostream, const char* location, unsigned index, const char* locationValue) const;
  AWS_REDSHIFT_API void OutputToStream(Aws::OStream& oStream, const char* location) const;

  ///@{
  /**
   * <p>The name of the parameter.</p>
   */
  inline const Aws::String& GetParameterName() const { return m_parameterName; }
  inline bool ParameterNameHasBeenSet() const { return m_parameterNameHasBeenSet; }
  template <typename ParameterNameT = Aws::String>
  void SetParameterName(ParameterNameT&& value) {
    m_parameterNameHasBeenSet = true;
    m_parameterName = std::forward<ParameterNameT>(value);
  }
  template <typename ParameterNameT = Aws::String>
  Parameter& WithParameterName(ParameterNameT&& value) {
    SetParameterName(std::forward<ParameterNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The value of the parameter. If <code>ParameterName</code> is
   * <code>wlm_json_configuration</code>, then the maximum size of
   * <code>ParameterValue</code> is 8000 characters.</p>
   */
  inline const Aws::String& GetParameterValue() const { return m_parameterValue; }
  inline bool ParameterValueHasBeenSet() const { return m_parameterValueHasBeenSet; }
  template <typename ParameterValueT = Aws::String>
  void SetParameterValue(ParameterValueT&& value) {
    m_parameterValueHasBeenSet = true;
    m_parameterValue = std::forward<ParameterValueT>(value);
  }
  template <typename ParameterValueT = Aws::String>
  Parameter& WithParameterValue(ParameterValueT&& value) {
    SetParameterValue(std::forward<ParameterValueT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A description of the parameter.</p>
   */
  inline const Aws::String& GetDescription() const { return m_description; }
  inline bool DescriptionHasBeenSet() const { return m_descriptionHasBeenSet; }
  template <typename DescriptionT = Aws::String>
  void SetDescription(DescriptionT&& value) {
    m_descriptionHasBeenSet = true;
    m_description = std::forward<DescriptionT>(value);
  }
  template <typename DescriptionT = Aws::String>
  Parameter& WithDescription(DescriptionT&& value) {
    SetDescription(std::forward<DescriptionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The source of the parameter value, such as "engine-default" or "user".</p>
   */
  inline const Aws::String& GetSource() const { return m_source; }
  inline bool SourceHasBeenSet() const { return m_sourceHasBeenSet; }
  template <typename SourceT = Aws::String>
  void SetSource(SourceT&& value) {
    m_sourceHasBeenSet = true;
    m_source = std::forward<SourceT>(value);
  }
  template <typename SourceT = Aws::String>
  Parameter& WithSource(SourceT&& value) {
    SetSource(std::forward<SourceT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The data type of the parameter.</p>
   */
  inline const Aws::String& GetDataType() const { return m_dataType; }
  inline bool DataTypeHasBeenSet() const { return m_dataTypeHasBeenSet; }
  template <typename DataTypeT = Aws::String>
  void SetDataType(DataTypeT&& value) {
    m_dataTypeHasBeenSet = true;
    m_dataType = std::forward<DataTypeT>(value);
  }
  template <typename DataTypeT = Aws::String>
  Parameter& WithDataType(DataTypeT&& value) {
    SetDataType(std::forward<DataTypeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The valid range of values for the parameter.</p>
   */
  inline const Aws::String& GetAllowedValues() const { return m_allowedValues; }
  inline bool AllowedValuesHasBeenSet() const { return m_allowedValuesHasBeenSet; }
  template <typename AllowedValuesT = Aws::String>
  void SetAllowedValues(AllowedValuesT&& value) {
    m_allowedValuesHasBeenSet = true;
    m_allowedValues = std::forward<AllowedValuesT>(value);
  }
  template <typename AllowedValuesT = Aws::String>
  Parameter& WithAllowedValues(AllowedValuesT&& value) {
    SetAllowedValues(std::forward<AllowedValuesT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies how to apply the WLM configuration parameter. Some properties can
   * be applied dynamically, while other properties require that any associated
   * clusters be rebooted for the configuration changes to be applied. For more
   * information about parameters and parameter groups, go to <a
   * href="https://docs.aws.amazon.com/redshift/latest/mgmt/working-with-parameter-groups.html">Amazon
   * Redshift Parameter Groups</a> in the <i>Amazon Redshift Cluster Management
   * Guide</i>.</p>
   */
  inline ParameterApplyType GetApplyType() const { return m_applyType; }
  inline bool ApplyTypeHasBeenSet() const { return m_applyTypeHasBeenSet; }
  inline void SetApplyType(ParameterApplyType value) {
    m_applyTypeHasBeenSet = true;
    m_applyType = value;
  }
  inline Parameter& WithApplyType(ParameterApplyType value) {
    SetApplyType(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>If <code>true</code>, the parameter can be modified. Some parameters have
   * security or operational implications that prevent them from being changed. </p>
   */
  inline bool GetIsModifiable() const { return m_isModifiable; }
  inline bool IsModifiableHasBeenSet() const { return m_isModifiableHasBeenSet; }
  inline void SetIsModifiable(bool value) {
    m_isModifiableHasBeenSet = true;
    m_isModifiable = value;
  }
  inline Parameter& WithIsModifiable(bool value) {
    SetIsModifiable(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The earliest engine version to which the parameter can apply.</p>
   */
  inline const Aws::String& GetMinimumEngineVersion() const { return m_minimumEngineVersion; }
  inline bool MinimumEngineVersionHasBeenSet() const { return m_minimumEngineVersionHasBeenSet; }
  template <typename MinimumEngineVersionT = Aws::String>
  void SetMinimumEngineVersion(MinimumEngineVersionT&& value) {
    m_minimumEngineVersionHasBeenSet = true;
    m_minimumEngineVersion = std::forward<MinimumEngineVersionT>(value);
  }
  template <typename MinimumEngineVersionT = Aws::String>
  Parameter& WithMinimumEngineVersion(MinimumEngineVersionT&& value) {
    SetMinimumEngineVersion(std::forward<MinimumEngineVersionT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_parameterName;
  bool m_parameterNameHasBeenSet = false;

  Aws::String m_parameterValue;
  bool m_parameterValueHasBeenSet = false;

  Aws::String m_description;
  bool m_descriptionHasBeenSet = false;

  Aws::String m_source;
  bool m_sourceHasBeenSet = false;

  Aws::String m_dataType;
  bool m_dataTypeHasBeenSet = false;

  Aws::String m_allowedValues;
  bool m_allowedValuesHasBeenSet = false;

  ParameterApplyType m_applyType{ParameterApplyType::NOT_SET};
  bool m_applyTypeHasBeenSet = false;

  bool m_isModifiable{false};
  bool m_isModifiableHasBeenSet = false;

  Aws::String m_minimumEngineVersion;
  bool m_minimumEngineVersionHasBeenSet = false;
};

}  // namespace Model
}  // namespace Redshift
}  // namespace Aws
