﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/resiliencehub/ResilienceHubRequest.h>
#include <aws/resiliencehub/ResilienceHub_EXPORTS.h>
#include <aws/resiliencehub/model/RejectGroupingRecommendationEntry.h>

#include <utility>

namespace Aws {
namespace ResilienceHub {
namespace Model {

/**
 */
class RejectResourceGroupingRecommendationsRequest : public ResilienceHubRequest {
 public:
  AWS_RESILIENCEHUB_API RejectResourceGroupingRecommendationsRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "RejectResourceGroupingRecommendations"; }

  AWS_RESILIENCEHUB_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>Amazon Resource Name (ARN) of the Resilience Hub application. The format for
   * this ARN is:
   * arn:<code>partition</code>:resiliencehub:<code>region</code>:<code>account</code>:app/<code>app-id</code>.
   * For more information about ARNs, see <a
   * href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">
   * Amazon Resource Names (ARNs)</a> in the <i>Amazon Web Services General
   * Reference</i> guide.</p>
   */
  inline const Aws::String& GetAppArn() const { return m_appArn; }
  inline bool AppArnHasBeenSet() const { return m_appArnHasBeenSet; }
  template <typename AppArnT = Aws::String>
  void SetAppArn(AppArnT&& value) {
    m_appArnHasBeenSet = true;
    m_appArn = std::forward<AppArnT>(value);
  }
  template <typename AppArnT = Aws::String>
  RejectResourceGroupingRecommendationsRequest& WithAppArn(AppArnT&& value) {
    SetAppArn(std::forward<AppArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>List of resource grouping recommendations you have selected to exclude from
   * your application.</p>
   */
  inline const Aws::Vector<RejectGroupingRecommendationEntry>& GetEntries() const { return m_entries; }
  inline bool EntriesHasBeenSet() const { return m_entriesHasBeenSet; }
  template <typename EntriesT = Aws::Vector<RejectGroupingRecommendationEntry>>
  void SetEntries(EntriesT&& value) {
    m_entriesHasBeenSet = true;
    m_entries = std::forward<EntriesT>(value);
  }
  template <typename EntriesT = Aws::Vector<RejectGroupingRecommendationEntry>>
  RejectResourceGroupingRecommendationsRequest& WithEntries(EntriesT&& value) {
    SetEntries(std::forward<EntriesT>(value));
    return *this;
  }
  template <typename EntriesT = RejectGroupingRecommendationEntry>
  RejectResourceGroupingRecommendationsRequest& AddEntries(EntriesT&& value) {
    m_entriesHasBeenSet = true;
    m_entries.emplace_back(std::forward<EntriesT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_appArn;
  bool m_appArnHasBeenSet = false;

  Aws::Vector<RejectGroupingRecommendationEntry> m_entries;
  bool m_entriesHasBeenSet = false;
};

}  // namespace Model
}  // namespace ResilienceHub
}  // namespace Aws
