﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/securityhub/SecurityHubRequest.h>
#include <aws/securityhub/SecurityHub_EXPORTS.h>
#include <aws/securityhub/model/AwsSecurityFindingFilters.h>
#include <aws/securityhub/model/SortCriterion.h>

#include <utility>

namespace Aws {
namespace SecurityHub {
namespace Model {

/**
 */
class GetFindingsRequest : public SecurityHubRequest {
 public:
  AWS_SECURITYHUB_API GetFindingsRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "GetFindings"; }

  AWS_SECURITYHUB_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The finding attributes used to define a condition to filter the returned
   * findings.</p> <p>You can filter by up to 10 finding attributes. For each
   * attribute, you can provide up to 20 filter values.</p> <p>Note that in the
   * available filter fields, <code>WorkflowState</code> is deprecated. To search for
   * a finding based on its workflow status, use <code>WorkflowStatus</code>.</p>
   */
  inline const AwsSecurityFindingFilters& GetFilters() const { return m_filters; }
  inline bool FiltersHasBeenSet() const { return m_filtersHasBeenSet; }
  template <typename FiltersT = AwsSecurityFindingFilters>
  void SetFilters(FiltersT&& value) {
    m_filtersHasBeenSet = true;
    m_filters = std::forward<FiltersT>(value);
  }
  template <typename FiltersT = AwsSecurityFindingFilters>
  GetFindingsRequest& WithFilters(FiltersT&& value) {
    SetFilters(std::forward<FiltersT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The finding attributes used to sort the list of returned findings.</p>
   */
  inline const Aws::Vector<SortCriterion>& GetSortCriteria() const { return m_sortCriteria; }
  inline bool SortCriteriaHasBeenSet() const { return m_sortCriteriaHasBeenSet; }
  template <typename SortCriteriaT = Aws::Vector<SortCriterion>>
  void SetSortCriteria(SortCriteriaT&& value) {
    m_sortCriteriaHasBeenSet = true;
    m_sortCriteria = std::forward<SortCriteriaT>(value);
  }
  template <typename SortCriteriaT = Aws::Vector<SortCriterion>>
  GetFindingsRequest& WithSortCriteria(SortCriteriaT&& value) {
    SetSortCriteria(std::forward<SortCriteriaT>(value));
    return *this;
  }
  template <typename SortCriteriaT = SortCriterion>
  GetFindingsRequest& AddSortCriteria(SortCriteriaT&& value) {
    m_sortCriteriaHasBeenSet = true;
    m_sortCriteria.emplace_back(std::forward<SortCriteriaT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The token that is required for pagination. On your first call to the
   * <code>GetFindings</code> operation, set the value of this parameter to
   * <code>NULL</code>.</p> <p>For subsequent calls to the operation, to continue
   * listing data, set the value of this parameter to the value returned from the
   * previous response.</p>
   */
  inline const Aws::String& GetNextToken() const { return m_nextToken; }
  inline bool NextTokenHasBeenSet() const { return m_nextTokenHasBeenSet; }
  template <typename NextTokenT = Aws::String>
  void SetNextToken(NextTokenT&& value) {
    m_nextTokenHasBeenSet = true;
    m_nextToken = std::forward<NextTokenT>(value);
  }
  template <typename NextTokenT = Aws::String>
  GetFindingsRequest& WithNextToken(NextTokenT&& value) {
    SetNextToken(std::forward<NextTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The maximum number of findings to return.</p>
   */
  inline int GetMaxResults() const { return m_maxResults; }
  inline bool MaxResultsHasBeenSet() const { return m_maxResultsHasBeenSet; }
  inline void SetMaxResults(int value) {
    m_maxResultsHasBeenSet = true;
    m_maxResults = value;
  }
  inline GetFindingsRequest& WithMaxResults(int value) {
    SetMaxResults(value);
    return *this;
  }
  ///@}
 private:
  AwsSecurityFindingFilters m_filters;
  bool m_filtersHasBeenSet = false;

  Aws::Vector<SortCriterion> m_sortCriteria;
  bool m_sortCriteriaHasBeenSet = false;

  Aws::String m_nextToken;
  bool m_nextTokenHasBeenSet = false;

  int m_maxResults{0};
  bool m_maxResultsHasBeenSet = false;
};

}  // namespace Model
}  // namespace SecurityHub
}  // namespace Aws
