﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/sesv2/SESV2_EXPORTS.h>
#include <aws/sesv2/model/Attachment.h>
#include <aws/sesv2/model/EmailTemplateContent.h>
#include <aws/sesv2/model/MessageHeader.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Json {
class JsonValue;
class JsonView;
}  // namespace Json
}  // namespace Utils
namespace SESV2 {
namespace Model {

/**
 * <p>An object that defines the email template to use for an email message, and
 * the values to use for any message variables in that template. An <i>email
 * template</i> is a type of message template that contains content that you want
 * to reuse in email messages that you send. You can specifiy the email template by
 * providing the name or ARN of an <i>email template</i> previously saved in your
 * Amazon SES account or by providing the full template content.</p><p><h3>See
 * Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/sesv2-2019-09-27/Template">AWS API
 * Reference</a></p>
 */
class Template {
 public:
  AWS_SESV2_API Template() = default;
  AWS_SESV2_API Template(Aws::Utils::Json::JsonView jsonValue);
  AWS_SESV2_API Template& operator=(Aws::Utils::Json::JsonView jsonValue);
  AWS_SESV2_API Aws::Utils::Json::JsonValue Jsonize() const;

  ///@{
  /**
   * <p>The name of the template. You will refer to this name when you send email
   * using the <code>SendEmail</code> or <code>SendBulkEmail</code> operations. </p>
   */
  inline const Aws::String& GetTemplateName() const { return m_templateName; }
  inline bool TemplateNameHasBeenSet() const { return m_templateNameHasBeenSet; }
  template <typename TemplateNameT = Aws::String>
  void SetTemplateName(TemplateNameT&& value) {
    m_templateNameHasBeenSet = true;
    m_templateName = std::forward<TemplateNameT>(value);
  }
  template <typename TemplateNameT = Aws::String>
  Template& WithTemplateName(TemplateNameT&& value) {
    SetTemplateName(std::forward<TemplateNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the template.</p>
   */
  inline const Aws::String& GetTemplateArn() const { return m_templateArn; }
  inline bool TemplateArnHasBeenSet() const { return m_templateArnHasBeenSet; }
  template <typename TemplateArnT = Aws::String>
  void SetTemplateArn(TemplateArnT&& value) {
    m_templateArnHasBeenSet = true;
    m_templateArn = std::forward<TemplateArnT>(value);
  }
  template <typename TemplateArnT = Aws::String>
  Template& WithTemplateArn(TemplateArnT&& value) {
    SetTemplateArn(std::forward<TemplateArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The content of the template.</p>  <p>Amazon SES supports only simple
   * substitions when you send email using the <code>SendEmail</code> or
   * <code>SendBulkEmail</code> operations and you provide the full template content
   * in the request.</p>
   */
  inline const EmailTemplateContent& GetTemplateContent() const { return m_templateContent; }
  inline bool TemplateContentHasBeenSet() const { return m_templateContentHasBeenSet; }
  template <typename TemplateContentT = EmailTemplateContent>
  void SetTemplateContent(TemplateContentT&& value) {
    m_templateContentHasBeenSet = true;
    m_templateContent = std::forward<TemplateContentT>(value);
  }
  template <typename TemplateContentT = EmailTemplateContent>
  Template& WithTemplateContent(TemplateContentT&& value) {
    SetTemplateContent(std::forward<TemplateContentT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>An object that defines the values to use for message variables in the
   * template. This object is a set of key-value pairs. Each key defines a message
   * variable in the template. The corresponding value defines the value to use for
   * that variable.</p>
   */
  inline const Aws::String& GetTemplateData() const { return m_templateData; }
  inline bool TemplateDataHasBeenSet() const { return m_templateDataHasBeenSet; }
  template <typename TemplateDataT = Aws::String>
  void SetTemplateData(TemplateDataT&& value) {
    m_templateDataHasBeenSet = true;
    m_templateData = std::forward<TemplateDataT>(value);
  }
  template <typename TemplateDataT = Aws::String>
  Template& WithTemplateData(TemplateDataT&& value) {
    SetTemplateData(std::forward<TemplateDataT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The list of message headers that will be added to the email message.</p>
   */
  inline const Aws::Vector<MessageHeader>& GetHeaders() const { return m_headers; }
  inline bool HeadersHasBeenSet() const { return m_headersHasBeenSet; }
  template <typename HeadersT = Aws::Vector<MessageHeader>>
  void SetHeaders(HeadersT&& value) {
    m_headersHasBeenSet = true;
    m_headers = std::forward<HeadersT>(value);
  }
  template <typename HeadersT = Aws::Vector<MessageHeader>>
  Template& WithHeaders(HeadersT&& value) {
    SetHeaders(std::forward<HeadersT>(value));
    return *this;
  }
  template <typename HeadersT = MessageHeader>
  Template& AddHeaders(HeadersT&& value) {
    m_headersHasBeenSet = true;
    m_headers.emplace_back(std::forward<HeadersT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> The List of attachments to include in your email. All recipients will
   * receive the same attachments.</p>
   */
  inline const Aws::Vector<Attachment>& GetAttachments() const { return m_attachments; }
  inline bool AttachmentsHasBeenSet() const { return m_attachmentsHasBeenSet; }
  template <typename AttachmentsT = Aws::Vector<Attachment>>
  void SetAttachments(AttachmentsT&& value) {
    m_attachmentsHasBeenSet = true;
    m_attachments = std::forward<AttachmentsT>(value);
  }
  template <typename AttachmentsT = Aws::Vector<Attachment>>
  Template& WithAttachments(AttachmentsT&& value) {
    SetAttachments(std::forward<AttachmentsT>(value));
    return *this;
  }
  template <typename AttachmentsT = Attachment>
  Template& AddAttachments(AttachmentsT&& value) {
    m_attachmentsHasBeenSet = true;
    m_attachments.emplace_back(std::forward<AttachmentsT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_templateName;
  bool m_templateNameHasBeenSet = false;

  Aws::String m_templateArn;
  bool m_templateArnHasBeenSet = false;

  EmailTemplateContent m_templateContent;
  bool m_templateContentHasBeenSet = false;

  Aws::String m_templateData;
  bool m_templateDataHasBeenSet = false;

  Aws::Vector<MessageHeader> m_headers;
  bool m_headersHasBeenSet = false;

  Aws::Vector<Attachment> m_attachments;
  bool m_attachmentsHasBeenSet = false;
};

}  // namespace Model
}  // namespace SESV2
}  // namespace Aws
