﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/storagegateway/StorageGatewayRequest.h>
#include <aws/storagegateway/StorageGateway_EXPORTS.h>
#include <aws/storagegateway/model/Tag.h>

#include <utility>

namespace Aws {
namespace StorageGateway {
namespace Model {

/**
 */
class CreateCachediSCSIVolumeRequest : public StorageGatewayRequest {
 public:
  AWS_STORAGEGATEWAY_API CreateCachediSCSIVolumeRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateCachediSCSIVolume"; }

  AWS_STORAGEGATEWAY_API Aws::String SerializePayload() const override;

  AWS_STORAGEGATEWAY_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{

  inline const Aws::String& GetGatewayARN() const { return m_gatewayARN; }
  inline bool GatewayARNHasBeenSet() const { return m_gatewayARNHasBeenSet; }
  template <typename GatewayARNT = Aws::String>
  void SetGatewayARN(GatewayARNT&& value) {
    m_gatewayARNHasBeenSet = true;
    m_gatewayARN = std::forward<GatewayARNT>(value);
  }
  template <typename GatewayARNT = Aws::String>
  CreateCachediSCSIVolumeRequest& WithGatewayARN(GatewayARNT&& value) {
    SetGatewayARN(std::forward<GatewayARNT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The size of the volume in bytes.</p>
   */
  inline long long GetVolumeSizeInBytes() const { return m_volumeSizeInBytes; }
  inline bool VolumeSizeInBytesHasBeenSet() const { return m_volumeSizeInBytesHasBeenSet; }
  inline void SetVolumeSizeInBytes(long long value) {
    m_volumeSizeInBytesHasBeenSet = true;
    m_volumeSizeInBytes = value;
  }
  inline CreateCachediSCSIVolumeRequest& WithVolumeSizeInBytes(long long value) {
    SetVolumeSizeInBytes(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The snapshot ID (e.g. "snap-1122aabb") of the snapshot to restore as the new
   * cached volume. Specify this field if you want to create the iSCSI storage volume
   * from a snapshot; otherwise, do not include this field. To list snapshots for
   * your account use <a
   * href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/ApiReference-query-DescribeSnapshots.html">DescribeSnapshots</a>
   * in the <i>Amazon Elastic Compute Cloud API Reference</i>.</p>
   */
  inline const Aws::String& GetSnapshotId() const { return m_snapshotId; }
  inline bool SnapshotIdHasBeenSet() const { return m_snapshotIdHasBeenSet; }
  template <typename SnapshotIdT = Aws::String>
  void SetSnapshotId(SnapshotIdT&& value) {
    m_snapshotIdHasBeenSet = true;
    m_snapshotId = std::forward<SnapshotIdT>(value);
  }
  template <typename SnapshotIdT = Aws::String>
  CreateCachediSCSIVolumeRequest& WithSnapshotId(SnapshotIdT&& value) {
    SetSnapshotId(std::forward<SnapshotIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the iSCSI target used by an initiator to connect to a volume and
   * used as a suffix for the target ARN. For example, specifying
   * <code>TargetName</code> as <i>myvolume</i> results in the target ARN of
   * <code>arn:aws:storagegateway:us-east-2:111122223333:gateway/sgw-12A3456B/target/iqn.1997-05.com.amazon:myvolume</code>.
   * The target name must be unique across all volumes on a gateway.</p> <p>If you
   * don't specify a value, Storage Gateway uses the value that was previously used
   * for this volume as the new target name.</p>
   */
  inline const Aws::String& GetTargetName() const { return m_targetName; }
  inline bool TargetNameHasBeenSet() const { return m_targetNameHasBeenSet; }
  template <typename TargetNameT = Aws::String>
  void SetTargetName(TargetNameT&& value) {
    m_targetNameHasBeenSet = true;
    m_targetName = std::forward<TargetNameT>(value);
  }
  template <typename TargetNameT = Aws::String>
  CreateCachediSCSIVolumeRequest& WithTargetName(TargetNameT&& value) {
    SetTargetName(std::forward<TargetNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ARN for an existing volume. Specifying this ARN makes the new volume into
   * an exact copy of the specified existing volume's latest recovery point. The
   * <code>VolumeSizeInBytes</code> value for this new volume must be equal to or
   * larger than the size of the existing volume, in bytes.</p>
   */
  inline const Aws::String& GetSourceVolumeARN() const { return m_sourceVolumeARN; }
  inline bool SourceVolumeARNHasBeenSet() const { return m_sourceVolumeARNHasBeenSet; }
  template <typename SourceVolumeARNT = Aws::String>
  void SetSourceVolumeARN(SourceVolumeARNT&& value) {
    m_sourceVolumeARNHasBeenSet = true;
    m_sourceVolumeARN = std::forward<SourceVolumeARNT>(value);
  }
  template <typename SourceVolumeARNT = Aws::String>
  CreateCachediSCSIVolumeRequest& WithSourceVolumeARN(SourceVolumeARNT&& value) {
    SetSourceVolumeARN(std::forward<SourceVolumeARNT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The network interface of the gateway on which to expose the iSCSI target.
   * Accepts IPv4 and IPv6 addresses. Use <a>DescribeGatewayInformation</a> to get a
   * list of the network interfaces available on a gateway.</p> <p>Valid Values: A
   * valid IP address.</p>
   */
  inline const Aws::String& GetNetworkInterfaceId() const { return m_networkInterfaceId; }
  inline bool NetworkInterfaceIdHasBeenSet() const { return m_networkInterfaceIdHasBeenSet; }
  template <typename NetworkInterfaceIdT = Aws::String>
  void SetNetworkInterfaceId(NetworkInterfaceIdT&& value) {
    m_networkInterfaceIdHasBeenSet = true;
    m_networkInterfaceId = std::forward<NetworkInterfaceIdT>(value);
  }
  template <typename NetworkInterfaceIdT = Aws::String>
  CreateCachediSCSIVolumeRequest& WithNetworkInterfaceId(NetworkInterfaceIdT&& value) {
    SetNetworkInterfaceId(std::forward<NetworkInterfaceIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A unique identifier that you use to retry a request. If you retry a request,
   * use the same <code>ClientToken</code> you specified in the initial request.</p>
   */
  inline const Aws::String& GetClientToken() const { return m_clientToken; }
  inline bool ClientTokenHasBeenSet() const { return m_clientTokenHasBeenSet; }
  template <typename ClientTokenT = Aws::String>
  void SetClientToken(ClientTokenT&& value) {
    m_clientTokenHasBeenSet = true;
    m_clientToken = std::forward<ClientTokenT>(value);
  }
  template <typename ClientTokenT = Aws::String>
  CreateCachediSCSIVolumeRequest& WithClientToken(ClientTokenT&& value) {
    SetClientToken(std::forward<ClientTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Set to <code>true</code> to use Amazon S3 server-side encryption with your
   * own KMS key, or <code>false</code> to use a key managed by Amazon S3.
   * Optional.</p> <p>Valid Values: <code>true</code> | <code>false</code> </p>
   */
  inline bool GetKMSEncrypted() const { return m_kMSEncrypted; }
  inline bool KMSEncryptedHasBeenSet() const { return m_kMSEncryptedHasBeenSet; }
  inline void SetKMSEncrypted(bool value) {
    m_kMSEncryptedHasBeenSet = true;
    m_kMSEncrypted = value;
  }
  inline CreateCachediSCSIVolumeRequest& WithKMSEncrypted(bool value) {
    SetKMSEncrypted(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of a symmetric customer master key (CMK) used
   * for Amazon S3 server-side encryption. Storage Gateway does not support
   * asymmetric CMKs. This value can only be set when <code>KMSEncrypted</code> is
   * <code>true</code>. Optional.</p>
   */
  inline const Aws::String& GetKMSKey() const { return m_kMSKey; }
  inline bool KMSKeyHasBeenSet() const { return m_kMSKeyHasBeenSet; }
  template <typename KMSKeyT = Aws::String>
  void SetKMSKey(KMSKeyT&& value) {
    m_kMSKeyHasBeenSet = true;
    m_kMSKey = std::forward<KMSKeyT>(value);
  }
  template <typename KMSKeyT = Aws::String>
  CreateCachediSCSIVolumeRequest& WithKMSKey(KMSKeyT&& value) {
    SetKMSKey(std::forward<KMSKeyT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A list of up to 50 tags that you can assign to a cached volume. Each tag is a
   * key-value pair.</p>  <p>Valid characters for key and value are letters,
   * spaces, and numbers that you can represent in UTF-8 format, and the following
   * special characters: + - = . _ : / @. The maximum length of a tag's key is 128
   * characters, and the maximum length for a tag's value is 256 characters.</p>
   *
   */
  inline const Aws::Vector<Tag>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Vector<Tag>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Vector<Tag>>
  CreateCachediSCSIVolumeRequest& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsT = Tag>
  CreateCachediSCSIVolumeRequest& AddTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace_back(std::forward<TagsT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_gatewayARN;
  bool m_gatewayARNHasBeenSet = false;

  long long m_volumeSizeInBytes{0};
  bool m_volumeSizeInBytesHasBeenSet = false;

  Aws::String m_snapshotId;
  bool m_snapshotIdHasBeenSet = false;

  Aws::String m_targetName;
  bool m_targetNameHasBeenSet = false;

  Aws::String m_sourceVolumeARN;
  bool m_sourceVolumeARNHasBeenSet = false;

  Aws::String m_networkInterfaceId;
  bool m_networkInterfaceIdHasBeenSet = false;

  Aws::String m_clientToken;
  bool m_clientTokenHasBeenSet = false;

  bool m_kMSEncrypted{false};
  bool m_kMSEncryptedHasBeenSet = false;

  Aws::String m_kMSKey;
  bool m_kMSKeyHasBeenSet = false;

  Aws::Vector<Tag> m_tags;
  bool m_tagsHasBeenSet = false;
};

}  // namespace Model
}  // namespace StorageGateway
}  // namespace Aws
