﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/UUID.h>
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/vpc-lattice/VPCLatticeRequest.h>
#include <aws/vpc-lattice/VPCLattice_EXPORTS.h>
#include <aws/vpc-lattice/model/RuleAction.h>
#include <aws/vpc-lattice/model/RuleMatch.h>

#include <utility>

namespace Aws {
namespace VPCLattice {
namespace Model {

/**
 */
class CreateRuleRequest : public VPCLatticeRequest {
 public:
  AWS_VPCLATTICE_API CreateRuleRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateRule"; }

  AWS_VPCLATTICE_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The ID or ARN of the service.</p>
   */
  inline const Aws::String& GetServiceIdentifier() const { return m_serviceIdentifier; }
  inline bool ServiceIdentifierHasBeenSet() const { return m_serviceIdentifierHasBeenSet; }
  template <typename ServiceIdentifierT = Aws::String>
  void SetServiceIdentifier(ServiceIdentifierT&& value) {
    m_serviceIdentifierHasBeenSet = true;
    m_serviceIdentifier = std::forward<ServiceIdentifierT>(value);
  }
  template <typename ServiceIdentifierT = Aws::String>
  CreateRuleRequest& WithServiceIdentifier(ServiceIdentifierT&& value) {
    SetServiceIdentifier(std::forward<ServiceIdentifierT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ID or ARN of the listener.</p>
   */
  inline const Aws::String& GetListenerIdentifier() const { return m_listenerIdentifier; }
  inline bool ListenerIdentifierHasBeenSet() const { return m_listenerIdentifierHasBeenSet; }
  template <typename ListenerIdentifierT = Aws::String>
  void SetListenerIdentifier(ListenerIdentifierT&& value) {
    m_listenerIdentifierHasBeenSet = true;
    m_listenerIdentifier = std::forward<ListenerIdentifierT>(value);
  }
  template <typename ListenerIdentifierT = Aws::String>
  CreateRuleRequest& WithListenerIdentifier(ListenerIdentifierT&& value) {
    SetListenerIdentifier(std::forward<ListenerIdentifierT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the rule. The name must be unique within the listener. The valid
   * characters are a-z, 0-9, and hyphens (-). You can't use a hyphen as the first or
   * last character, or immediately after another hyphen.</p>
   */
  inline const Aws::String& GetName() const { return m_name; }
  inline bool NameHasBeenSet() const { return m_nameHasBeenSet; }
  template <typename NameT = Aws::String>
  void SetName(NameT&& value) {
    m_nameHasBeenSet = true;
    m_name = std::forward<NameT>(value);
  }
  template <typename NameT = Aws::String>
  CreateRuleRequest& WithName(NameT&& value) {
    SetName(std::forward<NameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The rule match.</p>
   */
  inline const RuleMatch& GetMatch() const { return m_match; }
  inline bool MatchHasBeenSet() const { return m_matchHasBeenSet; }
  template <typename MatchT = RuleMatch>
  void SetMatch(MatchT&& value) {
    m_matchHasBeenSet = true;
    m_match = std::forward<MatchT>(value);
  }
  template <typename MatchT = RuleMatch>
  CreateRuleRequest& WithMatch(MatchT&& value) {
    SetMatch(std::forward<MatchT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The priority assigned to the rule. Each rule for a specific listener must
   * have a unique priority. The lower the priority number the higher the
   * priority.</p>
   */
  inline int GetPriority() const { return m_priority; }
  inline bool PriorityHasBeenSet() const { return m_priorityHasBeenSet; }
  inline void SetPriority(int value) {
    m_priorityHasBeenSet = true;
    m_priority = value;
  }
  inline CreateRuleRequest& WithPriority(int value) {
    SetPriority(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The action for the default rule.</p>
   */
  inline const RuleAction& GetAction() const { return m_action; }
  inline bool ActionHasBeenSet() const { return m_actionHasBeenSet; }
  template <typename ActionT = RuleAction>
  void SetAction(ActionT&& value) {
    m_actionHasBeenSet = true;
    m_action = std::forward<ActionT>(value);
  }
  template <typename ActionT = RuleAction>
  CreateRuleRequest& WithAction(ActionT&& value) {
    SetAction(std::forward<ActionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A unique, case-sensitive identifier that you provide to ensure the
   * idempotency of the request. If you retry a request that completed successfully
   * using the same client token and parameters, the retry succeeds without
   * performing any actions. If the parameters aren't identical, the retry fails.</p>
   */
  inline const Aws::String& GetClientToken() const { return m_clientToken; }
  inline bool ClientTokenHasBeenSet() const { return m_clientTokenHasBeenSet; }
  template <typename ClientTokenT = Aws::String>
  void SetClientToken(ClientTokenT&& value) {
    m_clientTokenHasBeenSet = true;
    m_clientToken = std::forward<ClientTokenT>(value);
  }
  template <typename ClientTokenT = Aws::String>
  CreateRuleRequest& WithClientToken(ClientTokenT&& value) {
    SetClientToken(std::forward<ClientTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The tags for the rule.</p>
   */
  inline const Aws::Map<Aws::String, Aws::String>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  CreateRuleRequest& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsKeyT = Aws::String, typename TagsValueT = Aws::String>
  CreateRuleRequest& AddTags(TagsKeyT&& key, TagsValueT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace(std::forward<TagsKeyT>(key), std::forward<TagsValueT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_serviceIdentifier;
  bool m_serviceIdentifierHasBeenSet = false;

  Aws::String m_listenerIdentifier;
  bool m_listenerIdentifierHasBeenSet = false;

  Aws::String m_name;
  bool m_nameHasBeenSet = false;

  RuleMatch m_match;
  bool m_matchHasBeenSet = false;

  int m_priority{0};
  bool m_priorityHasBeenSet = false;

  RuleAction m_action;
  bool m_actionHasBeenSet = false;

  Aws::String m_clientToken{Aws::Utils::UUID::PseudoRandomUUID()};
  bool m_clientTokenHasBeenSet = true;

  Aws::Map<Aws::String, Aws::String> m_tags;
  bool m_tagsHasBeenSet = false;
};

}  // namespace Model
}  // namespace VPCLattice
}  // namespace Aws
