﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/xray/XRayRequest.h>
#include <aws/xray/XRay_EXPORTS.h>
#include <aws/xray/model/Tag.h>

#include <utility>

namespace Aws {
namespace XRay {
namespace Model {

/**
 */
class TagResourceRequest : public XRayRequest {
 public:
  AWS_XRAY_API TagResourceRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "TagResource"; }

  AWS_XRAY_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The Amazon Resource Number (ARN) of an X-Ray group or sampling rule.</p>
   */
  inline const Aws::String& GetResourceARN() const { return m_resourceARN; }
  inline bool ResourceARNHasBeenSet() const { return m_resourceARNHasBeenSet; }
  template <typename ResourceARNT = Aws::String>
  void SetResourceARN(ResourceARNT&& value) {
    m_resourceARNHasBeenSet = true;
    m_resourceARN = std::forward<ResourceARNT>(value);
  }
  template <typename ResourceARNT = Aws::String>
  TagResourceRequest& WithResourceARN(ResourceARNT&& value) {
    SetResourceARN(std::forward<ResourceARNT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A map that contains one or more tag keys and tag values to attach to an X-Ray
   * group or sampling rule. For more information about ways to use tags, see <a
   * href="https://docs.aws.amazon.com/general/latest/gr/aws_tagging.html">Tagging
   * Amazon Web Services resources</a> in the <i>Amazon Web Services General
   * Reference</i>.</p> <p>The following restrictions apply to tags:</p> <ul> <li>
   * <p>Maximum number of user-applied tags per resource: 50</p> </li> <li>
   * <p>Maximum tag key length: 128 Unicode characters</p> </li> <li> <p>Maximum tag
   * value length: 256 Unicode characters</p> </li> <li> <p>Valid values for key and
   * value: a-z, A-Z, 0-9, space, and the following characters: _ . : / = + - and
   * @</p> </li> <li> <p>Tag keys and values are case sensitive.</p> </li> <li>
   * <p>Don't use <code>aws:</code> as a prefix for keys; it's reserved for Amazon
   * Web Services use. You cannot edit or delete system tags.</p> </li> </ul>
   */
  inline const Aws::Vector<Tag>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Vector<Tag>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Vector<Tag>>
  TagResourceRequest& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsT = Tag>
  TagResourceRequest& AddTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace_back(std::forward<TagsT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_resourceARN;
  bool m_resourceARNHasBeenSet = false;

  Aws::Vector<Tag> m_tags;
  bool m_tagsHasBeenSet = false;
};

}  // namespace Model
}  // namespace XRay
}  // namespace Aws
