﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/firehose/Firehose_EXPORTS.h>
#include <aws/firehose/model/BufferingHints.h>
#include <aws/firehose/model/CloudWatchLoggingOptions.h>
#include <aws/firehose/model/CompressionFormat.h>
#include <aws/firehose/model/EncryptionConfiguration.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Json {
class JsonValue;
class JsonView;
}  // namespace Json
}  // namespace Utils
namespace Firehose {
namespace Model {

/**
 * <p>Describes a destination in Amazon S3.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/firehose-2015-08-04/S3DestinationDescription">AWS
 * API Reference</a></p>
 */
class S3DestinationDescription {
 public:
  AWS_FIREHOSE_API S3DestinationDescription() = default;
  AWS_FIREHOSE_API S3DestinationDescription(Aws::Utils::Json::JsonView jsonValue);
  AWS_FIREHOSE_API S3DestinationDescription& operator=(Aws::Utils::Json::JsonView jsonValue);
  AWS_FIREHOSE_API Aws::Utils::Json::JsonValue Jsonize() const;

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the Amazon Web Services credentials. For
   * more information, see <a
   * href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon
   * Resource Names (ARNs) and Amazon Web Services Service Namespaces</a>.</p>
   */
  inline const Aws::String& GetRoleARN() const { return m_roleARN; }
  inline bool RoleARNHasBeenSet() const { return m_roleARNHasBeenSet; }
  template <typename RoleARNT = Aws::String>
  void SetRoleARN(RoleARNT&& value) {
    m_roleARNHasBeenSet = true;
    m_roleARN = std::forward<RoleARNT>(value);
  }
  template <typename RoleARNT = Aws::String>
  S3DestinationDescription& WithRoleARN(RoleARNT&& value) {
    SetRoleARN(std::forward<RoleARNT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ARN of the S3 bucket. For more information, see <a
   * href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon
   * Resource Names (ARNs) and Amazon Web Services Service Namespaces</a>.</p>
   */
  inline const Aws::String& GetBucketARN() const { return m_bucketARN; }
  inline bool BucketARNHasBeenSet() const { return m_bucketARNHasBeenSet; }
  template <typename BucketARNT = Aws::String>
  void SetBucketARN(BucketARNT&& value) {
    m_bucketARNHasBeenSet = true;
    m_bucketARN = std::forward<BucketARNT>(value);
  }
  template <typename BucketARNT = Aws::String>
  S3DestinationDescription& WithBucketARN(BucketARNT&& value) {
    SetBucketARN(std::forward<BucketARNT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The "YYYY/MM/DD/HH" time format prefix is automatically used for delivered
   * Amazon S3 files. You can also specify a custom prefix, as described in <a
   * href="https://docs.aws.amazon.com/firehose/latest/dev/s3-prefixes.html">Custom
   * Prefixes for Amazon S3 Objects</a>.</p>
   */
  inline const Aws::String& GetPrefix() const { return m_prefix; }
  inline bool PrefixHasBeenSet() const { return m_prefixHasBeenSet; }
  template <typename PrefixT = Aws::String>
  void SetPrefix(PrefixT&& value) {
    m_prefixHasBeenSet = true;
    m_prefix = std::forward<PrefixT>(value);
  }
  template <typename PrefixT = Aws::String>
  S3DestinationDescription& WithPrefix(PrefixT&& value) {
    SetPrefix(std::forward<PrefixT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A prefix that Firehose evaluates and adds to failed records before writing
   * them to S3. This prefix appears immediately following the bucket name. For
   * information about how to specify this prefix, see <a
   * href="https://docs.aws.amazon.com/firehose/latest/dev/s3-prefixes.html">Custom
   * Prefixes for Amazon S3 Objects</a>.</p>
   */
  inline const Aws::String& GetErrorOutputPrefix() const { return m_errorOutputPrefix; }
  inline bool ErrorOutputPrefixHasBeenSet() const { return m_errorOutputPrefixHasBeenSet; }
  template <typename ErrorOutputPrefixT = Aws::String>
  void SetErrorOutputPrefix(ErrorOutputPrefixT&& value) {
    m_errorOutputPrefixHasBeenSet = true;
    m_errorOutputPrefix = std::forward<ErrorOutputPrefixT>(value);
  }
  template <typename ErrorOutputPrefixT = Aws::String>
  S3DestinationDescription& WithErrorOutputPrefix(ErrorOutputPrefixT&& value) {
    SetErrorOutputPrefix(std::forward<ErrorOutputPrefixT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The buffering option. If no value is specified, <code>BufferingHints</code>
   * object default values are used.</p>
   */
  inline const BufferingHints& GetBufferingHints() const { return m_bufferingHints; }
  inline bool BufferingHintsHasBeenSet() const { return m_bufferingHintsHasBeenSet; }
  template <typename BufferingHintsT = BufferingHints>
  void SetBufferingHints(BufferingHintsT&& value) {
    m_bufferingHintsHasBeenSet = true;
    m_bufferingHints = std::forward<BufferingHintsT>(value);
  }
  template <typename BufferingHintsT = BufferingHints>
  S3DestinationDescription& WithBufferingHints(BufferingHintsT&& value) {
    SetBufferingHints(std::forward<BufferingHintsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The compression format. If no value is specified, the default is
   * <code>UNCOMPRESSED</code>.</p>
   */
  inline CompressionFormat GetCompressionFormat() const { return m_compressionFormat; }
  inline bool CompressionFormatHasBeenSet() const { return m_compressionFormatHasBeenSet; }
  inline void SetCompressionFormat(CompressionFormat value) {
    m_compressionFormatHasBeenSet = true;
    m_compressionFormat = value;
  }
  inline S3DestinationDescription& WithCompressionFormat(CompressionFormat value) {
    SetCompressionFormat(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The encryption configuration. If no value is specified, the default is no
   * encryption.</p>
   */
  inline const EncryptionConfiguration& GetEncryptionConfiguration() const { return m_encryptionConfiguration; }
  inline bool EncryptionConfigurationHasBeenSet() const { return m_encryptionConfigurationHasBeenSet; }
  template <typename EncryptionConfigurationT = EncryptionConfiguration>
  void SetEncryptionConfiguration(EncryptionConfigurationT&& value) {
    m_encryptionConfigurationHasBeenSet = true;
    m_encryptionConfiguration = std::forward<EncryptionConfigurationT>(value);
  }
  template <typename EncryptionConfigurationT = EncryptionConfiguration>
  S3DestinationDescription& WithEncryptionConfiguration(EncryptionConfigurationT&& value) {
    SetEncryptionConfiguration(std::forward<EncryptionConfigurationT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon CloudWatch logging options for your Firehose stream.</p>
   */
  inline const CloudWatchLoggingOptions& GetCloudWatchLoggingOptions() const { return m_cloudWatchLoggingOptions; }
  inline bool CloudWatchLoggingOptionsHasBeenSet() const { return m_cloudWatchLoggingOptionsHasBeenSet; }
  template <typename CloudWatchLoggingOptionsT = CloudWatchLoggingOptions>
  void SetCloudWatchLoggingOptions(CloudWatchLoggingOptionsT&& value) {
    m_cloudWatchLoggingOptionsHasBeenSet = true;
    m_cloudWatchLoggingOptions = std::forward<CloudWatchLoggingOptionsT>(value);
  }
  template <typename CloudWatchLoggingOptionsT = CloudWatchLoggingOptions>
  S3DestinationDescription& WithCloudWatchLoggingOptions(CloudWatchLoggingOptionsT&& value) {
    SetCloudWatchLoggingOptions(std::forward<CloudWatchLoggingOptionsT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_roleARN;
  bool m_roleARNHasBeenSet = false;

  Aws::String m_bucketARN;
  bool m_bucketARNHasBeenSet = false;

  Aws::String m_prefix;
  bool m_prefixHasBeenSet = false;

  Aws::String m_errorOutputPrefix;
  bool m_errorOutputPrefixHasBeenSet = false;

  BufferingHints m_bufferingHints;
  bool m_bufferingHintsHasBeenSet = false;

  CompressionFormat m_compressionFormat{CompressionFormat::NOT_SET};
  bool m_compressionFormatHasBeenSet = false;

  EncryptionConfiguration m_encryptionConfiguration;
  bool m_encryptionConfigurationHasBeenSet = false;

  CloudWatchLoggingOptions m_cloudWatchLoggingOptions;
  bool m_cloudWatchLoggingOptionsHasBeenSet = false;
};

}  // namespace Model
}  // namespace Firehose
}  // namespace Aws
