﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/s3/S3_EXPORTS.h>
#include <aws/s3/model/CopyObjectResultDetails.h>
#include <aws/s3/model/RequestCharged.h>
#include <aws/s3/model/ServerSideEncryption.h>

#include <utility>

namespace Aws {
template <typename RESULT_TYPE>
class AmazonWebServiceResult;

namespace Utils {
namespace Xml {
class XmlDocument;
}  // namespace Xml
}  // namespace Utils
namespace S3 {
namespace Model {
class CopyObjectResult {
 public:
  AWS_S3_API CopyObjectResult() = default;
  AWS_S3_API CopyObjectResult(const Aws::AmazonWebServiceResult<Aws::Utils::Xml::XmlDocument>& result);
  AWS_S3_API CopyObjectResult& operator=(const Aws::AmazonWebServiceResult<Aws::Utils::Xml::XmlDocument>& result);

  ///@{
  /**
   * <p>If the object expiration is configured, the response includes this
   * header.</p>  <p>Object expiration information is not returned in directory
   * buckets and this header returns the value "<code>NotImplemented</code>" in all
   * responses for directory buckets.</p>
   */
  inline const Aws::String& GetExpiration() const { return m_expiration; }
  template <typename ExpirationT = Aws::String>
  void SetExpiration(ExpirationT&& value) {
    m_expirationHasBeenSet = true;
    m_expiration = std::forward<ExpirationT>(value);
  }
  template <typename ExpirationT = Aws::String>
  CopyObjectResult& WithExpiration(ExpirationT&& value) {
    SetExpiration(std::forward<ExpirationT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Version ID of the source object that was copied.</p>  <p>This
   * functionality is not supported when the source object is in a directory
   * bucket.</p>
   */
  inline const Aws::String& GetCopySourceVersionId() const { return m_copySourceVersionId; }
  template <typename CopySourceVersionIdT = Aws::String>
  void SetCopySourceVersionId(CopySourceVersionIdT&& value) {
    m_copySourceVersionIdHasBeenSet = true;
    m_copySourceVersionId = std::forward<CopySourceVersionIdT>(value);
  }
  template <typename CopySourceVersionIdT = Aws::String>
  CopyObjectResult& WithCopySourceVersionId(CopySourceVersionIdT&& value) {
    SetCopySourceVersionId(std::forward<CopySourceVersionIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Version ID of the newly created copy.</p>  <p>This functionality is not
   * supported for directory buckets.</p>
   */
  inline const Aws::String& GetVersionId() const { return m_versionId; }
  template <typename VersionIdT = Aws::String>
  void SetVersionId(VersionIdT&& value) {
    m_versionIdHasBeenSet = true;
    m_versionId = std::forward<VersionIdT>(value);
  }
  template <typename VersionIdT = Aws::String>
  CopyObjectResult& WithVersionId(VersionIdT&& value) {
    SetVersionId(std::forward<VersionIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The server-side encryption algorithm used when you store this object in
   * Amazon S3 or Amazon FSx.</p>  <p>When accessing data stored in Amazon FSx
   * file systems using S3 access points, the only valid server side encryption
   * option is <code>aws:fsx</code>.</p>
   */
  inline ServerSideEncryption GetServerSideEncryption() const { return m_serverSideEncryption; }
  inline void SetServerSideEncryption(ServerSideEncryption value) {
    m_serverSideEncryptionHasBeenSet = true;
    m_serverSideEncryption = value;
  }
  inline CopyObjectResult& WithServerSideEncryption(ServerSideEncryption value) {
    SetServerSideEncryption(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>If server-side encryption with a customer-provided encryption key was
   * requested, the response will include this header to confirm the encryption
   * algorithm that's used.</p>  <p>This functionality is not supported for
   * directory buckets.</p>
   */
  inline const Aws::String& GetSSECustomerAlgorithm() const { return m_sSECustomerAlgorithm; }
  template <typename SSECustomerAlgorithmT = Aws::String>
  void SetSSECustomerAlgorithm(SSECustomerAlgorithmT&& value) {
    m_sSECustomerAlgorithmHasBeenSet = true;
    m_sSECustomerAlgorithm = std::forward<SSECustomerAlgorithmT>(value);
  }
  template <typename SSECustomerAlgorithmT = Aws::String>
  CopyObjectResult& WithSSECustomerAlgorithm(SSECustomerAlgorithmT&& value) {
    SetSSECustomerAlgorithm(std::forward<SSECustomerAlgorithmT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>If server-side encryption with a customer-provided encryption key was
   * requested, the response will include this header to provide the round-trip
   * message integrity verification of the customer-provided encryption key.</p>
   *  <p>This functionality is not supported for directory buckets.</p>
   */
  inline const Aws::String& GetSSECustomerKeyMD5() const { return m_sSECustomerKeyMD5; }
  template <typename SSECustomerKeyMD5T = Aws::String>
  void SetSSECustomerKeyMD5(SSECustomerKeyMD5T&& value) {
    m_sSECustomerKeyMD5HasBeenSet = true;
    m_sSECustomerKeyMD5 = std::forward<SSECustomerKeyMD5T>(value);
  }
  template <typename SSECustomerKeyMD5T = Aws::String>
  CopyObjectResult& WithSSECustomerKeyMD5(SSECustomerKeyMD5T&& value) {
    SetSSECustomerKeyMD5(std::forward<SSECustomerKeyMD5T>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>If present, indicates the ID of the KMS key that was used for object
   * encryption.</p>
   */
  inline const Aws::String& GetSSEKMSKeyId() const { return m_sSEKMSKeyId; }
  template <typename SSEKMSKeyIdT = Aws::String>
  void SetSSEKMSKeyId(SSEKMSKeyIdT&& value) {
    m_sSEKMSKeyIdHasBeenSet = true;
    m_sSEKMSKeyId = std::forward<SSEKMSKeyIdT>(value);
  }
  template <typename SSEKMSKeyIdT = Aws::String>
  CopyObjectResult& WithSSEKMSKeyId(SSEKMSKeyIdT&& value) {
    SetSSEKMSKeyId(std::forward<SSEKMSKeyIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>If present, indicates the Amazon Web Services KMS Encryption Context to use
   * for object encryption. The value of this header is a Base64 encoded UTF-8 string
   * holding JSON with the encryption context key-value pairs.</p>
   */
  inline const Aws::String& GetSSEKMSEncryptionContext() const { return m_sSEKMSEncryptionContext; }
  template <typename SSEKMSEncryptionContextT = Aws::String>
  void SetSSEKMSEncryptionContext(SSEKMSEncryptionContextT&& value) {
    m_sSEKMSEncryptionContextHasBeenSet = true;
    m_sSEKMSEncryptionContext = std::forward<SSEKMSEncryptionContextT>(value);
  }
  template <typename SSEKMSEncryptionContextT = Aws::String>
  CopyObjectResult& WithSSEKMSEncryptionContext(SSEKMSEncryptionContextT&& value) {
    SetSSEKMSEncryptionContext(std::forward<SSEKMSEncryptionContextT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Indicates whether the copied object uses an S3 Bucket Key for server-side
   * encryption with Key Management Service (KMS) keys (SSE-KMS).</p>
   */
  inline bool GetBucketKeyEnabled() const { return m_bucketKeyEnabled; }
  inline void SetBucketKeyEnabled(bool value) {
    m_bucketKeyEnabledHasBeenSet = true;
    m_bucketKeyEnabled = value;
  }
  inline CopyObjectResult& WithBucketKeyEnabled(bool value) {
    SetBucketKeyEnabled(value);
    return *this;
  }
  ///@}

  ///@{

  inline RequestCharged GetRequestCharged() const { return m_requestCharged; }
  inline void SetRequestCharged(RequestCharged value) {
    m_requestChargedHasBeenSet = true;
    m_requestCharged = value;
  }
  inline CopyObjectResult& WithRequestCharged(RequestCharged value) {
    SetRequestCharged(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Container for all response elements.</p>
   */
  inline const CopyObjectResultDetails& GetCopyObjectResultDetails() const { return m_copyObjectResultDetails; }
  template <typename CopyObjectResultDetailsT = CopyObjectResultDetails>
  void SetCopyObjectResultDetails(CopyObjectResultDetailsT&& value) {
    m_copyObjectResultDetailsHasBeenSet = true;
    m_copyObjectResultDetails = std::forward<CopyObjectResultDetailsT>(value);
  }
  template <typename CopyObjectResultDetailsT = CopyObjectResultDetails>
  CopyObjectResult& WithCopyObjectResultDetails(CopyObjectResultDetailsT&& value) {
    SetCopyObjectResultDetails(std::forward<CopyObjectResultDetailsT>(value));
    return *this;
  }
  ///@}

  ///@{

  inline const Aws::String& GetRequestId() const { return m_requestId; }
  template <typename RequestIdT = Aws::String>
  void SetRequestId(RequestIdT&& value) {
    m_requestIdHasBeenSet = true;
    m_requestId = std::forward<RequestIdT>(value);
  }
  template <typename RequestIdT = Aws::String>
  CopyObjectResult& WithRequestId(RequestIdT&& value) {
    SetRequestId(std::forward<RequestIdT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_expiration;
  bool m_expirationHasBeenSet = false;

  Aws::String m_copySourceVersionId;
  bool m_copySourceVersionIdHasBeenSet = false;

  Aws::String m_versionId;
  bool m_versionIdHasBeenSet = false;

  ServerSideEncryption m_serverSideEncryption{ServerSideEncryption::NOT_SET};
  bool m_serverSideEncryptionHasBeenSet = false;

  Aws::String m_sSECustomerAlgorithm;
  bool m_sSECustomerAlgorithmHasBeenSet = false;

  Aws::String m_sSECustomerKeyMD5;
  bool m_sSECustomerKeyMD5HasBeenSet = false;

  Aws::String m_sSEKMSKeyId;
  bool m_sSEKMSKeyIdHasBeenSet = false;

  Aws::String m_sSEKMSEncryptionContext;
  bool m_sSEKMSEncryptionContextHasBeenSet = false;

  bool m_bucketKeyEnabled{false};
  bool m_bucketKeyEnabledHasBeenSet = false;

  RequestCharged m_requestCharged{RequestCharged::NOT_SET};
  bool m_requestChargedHasBeenSet = false;

  CopyObjectResultDetails m_copyObjectResultDetails;
  bool m_copyObjectResultDetailsHasBeenSet = false;

  Aws::String m_requestId;
  bool m_requestIdHasBeenSet = false;
};

}  // namespace Model
}  // namespace S3
}  // namespace Aws
