#
# Copyright (C) 2022 Intel Corporation
#
# This software and the related documents are Intel copyrighted materials, and your use of them
# is governed by the express license under which they were provided to you ("License"). Unless
# the License provides otherwise, you may not use, modify, copy, publish, distribute, disclose
# or transmit this software or the related documents without Intel's prior written permission.
#
# This software and the related documents are provided as is, with no express or implied
# warranties, other than those that are expressly stated in the License.
#

# -----------------------------------------------------------------------------------------------------
# This example shows how to compare data in two Advisor projects with a specified matching confidence threshold.
# -----------------------------------------------------------------------------------------------------

import os
import sys

try:

    import advisor

except ImportError:

    print(
        """Import error: Python could not resolve path to Advisor's pythonapi directory.
        To fix, either manually add path to the pythonapi directory into PYTHONPATH environment
        variable, or use advixe-vars.* scripts to set up product environment variables automatically."""
    )
    sys.exit(1)

# Check command-line arguments.
if len(sys.argv) < 3:
    print('Usage: "python {} path_to_project_dir1 path_to_project_dir2"'.format(__file__))
    sys.exit(2)

# Open the Advisor Projects.
project_dirs = sys.argv[1:]
projects = []
for project_dir in project_dirs:
    projects.append(advisor.open_project(project_dir))

# Load the data in each Advisor Project then compare the survey results.
data1 = projects[0].load(advisor.SURVEY)
data2 = projects[1].load(advisor.SURVEY)

# Compare data from different profiles with a specified matching confidence threshold.
# The result of the comparison will contain pairs of matching loops/functions from different profiles
# with matching confidence not lower than the specified threshold.
result = data1.compare_ex(data2, 0.)

# Save the results of the comparison to a file on the disk.
with open("comparison_result_ex.txt", "w") as f:

    f.write(
        "\n{:>15}|{:>45}|{:>30}|{:>20}|{:>20}|{:>15}|{:>15}|{:>30}|{:>30}|{:>15}".format(
            "Loop ID",
            "Mangled Name",
            "Source",
            "Type",
            "Data Type",
            "Nest. level",
            "Max depth",
            "Avg. Trip Count",
            "Call count",
            "Confidence%",
        )
    )

    for info in result:

        row = info.row1
        f.write(
            "\n\n{:>15}|{:>45}|{:>30}|{:>20}|{:>20}|{:>15}|{:>15}|{:>30}|{:>30}|{:>15}%".format(
                row.loop_function_id,
                row.mangled_name[:45],
                row.source_location[:30],
                row.type[:20],
                row.data_types[:20],
                row.nesting_level,
                row.max_depth,
                row.average_trip_count,
                row.call_count[:30],
                info.conf,
            )
        )

        row = info.row2
        f.write(
            "\n{:>15}|{:>45}|{:>30}|{:>20}|{:>20}|{:>15}|{:>15}|{:>30}|{:>30}|".format(
                row.loop_function_id,
                row.mangled_name[:45],
                row.source_location[:30],
                row.type[:20],
                row.data_types[:20],
                row.nesting_level,
                row.max_depth,
                row.average_trip_count,
                row.call_count[:30],
            )
        )
