#
# Copyright (C) 2017 Intel Corporation
#
# This software and the related documents are Intel copyrighted materials, and your use of them
# is governed by the express license under which they were provided to you ("License"). Unless
# the License provides otherwise, you may not use, modify, copy, publish, distribute, disclose
# or transmit this software or the related documents without Intel's prior written permission.
#
# This software and the related documents are provided as is, with no express or implied
# warranties, other than those that are expressly stated in the License.
#


# ------------------------------------------------------------------------------
# This example shows how to create an HTML report that displays the data set in
# an Advisor project. The HTML report is an interactive table featuring
# columns, resizing, sorting, filtering, and rows collapsing.

# Two methods exist for creating the HTML report.
# advisor.html_report.from_rows(
#   data,
#   report_file='report.html',
#   columns=None,
#   groups=None,
#   minimal=False
# )
#
# advisor.html_report.from_json(
#   json_data,
#   report_file='report.html',
#   minimal=False
#  )
# ------------------------------------------------------------------------------

import os
import sys
import webbrowser

try:

    import advisor

except ImportError:

    print(
        """Import error: Python could not resolve path to Advisor's pythonapi directory.
        To fix, either manually add path to the pythonapi directory into PYTHONPATH environment
        variable, or use advixe-vars.* scripts to set up product environment variables automatically."""
    )
    sys.exit(1)

# Check command-line arguments.
if __name__ == "__main__":

    if len(sys.argv) < 2:
        print('Usage: "python {} path_to_project_dir [report_file_name]"'.format(__file__))
        sys.exit(2)

    # Open the Advisor Project and load the data.
    project = advisor.open_project(sys.argv[1])
    data = project.load(advisor.ALL)

    COLUMNS = [
        "function_call_sites_and_loops",
        "self_time",
        "optimization_details",
        "min_trip_count",
        "average_trip_count",
        "max_trip_count",
        "source_location",
    ]
    # To group the columns, use the optional 'groups' parameter, which has the following format:
    # [{ name: 'group',                           - the group name
    #    description(optional):'This is a group', - tooltip with group description
    #    start: 0,                                - the index of the first column in the group
    #    end: 4                                   - the index of the last column in the group
    # },
    # {...}]
    GROUPS = [
        {"name": "group1", "start": 0, "end": 2},
        {"name": "group2", "description": "group2", "start": 4, "end": 5},
    ]

    # To add some common metrics use the optional 'metrics' parameter, which is represented as the object with pairs: {key, value}
    METRICS = {  # caption is used to add caption to the Metrics table
        "caption": "Total Metrics",
        "Program Time": data.metrics.program_time,
        "Total GFLOPS": data.metrics.total_gflops,
        "Total AI": data.metrics.total_ai,
        "Total Non-zero GFLOPS": data.metrics.total_nonzero_gflops,
        "Total Non-zero AI": data.metrics.total_nonzero_ai,
    }

    # To add images to the additional section of the report use the optional 'additional_images' parameter, using the format shown in the example below
    # If you do not want to add a custom width and height, you can use the string format instead of the object format
    IMAGES = [
        {
            "width": "200px",
            "height": "100px",
            "base64": "data:image/png;base64,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",
        }
    ]

    report_file = sys.argv[2] if len(sys.argv) == 3 else "report.html"
    # Add '.html' extension if there is no extention.
    report_file, ext = os.path.splitext(report_file)
    report_file += ".html" if len(ext) == 0 else ext

    result = advisor.html_report.from_rows(
        data.bottomup,
        columns=COLUMNS,
        groups=GROUPS,
        metrics=METRICS,
        additional_images=IMAGES,
        report_file=report_file,
        minimal=False,
    )

    if not result:
        print("No data for HTML report...")
        sys.exit(3)

    # Remove the following lines to disable opening the report in a browser.
    if len(sys.argv) == 2:
        webbrowser.open(report_file)
