# Copyright (C) 2025 Intel Corporation
# SPDX-License-Identifier: MIT

import json
import logging

import jsonschema.exceptions
from jsonschema import validate
from abc import ABC, abstractmethod
from pathlib import Path
from typing import List, Dict

import defusedxml.ElementTree as ET

from mpp.core.types import MetricDefinition
from mpp.parsers.metric_fields import _XmlMetricFields, _JsonMetricFields
from tools.common.validators.input_validators import FileValidator

RETIRE_LATENCY_STRING = 'retire_latency'


def validate_file(file_path: Path):
    file_validator = FileValidator(file_must_exist=True, max_file_size=10 * 1024 * 1024)
    file_validator(str(file_path))


class JsonObjectValidator:
    """
    Validate json file
    """

    def __init__(self, file_spec: Path, schema: dict):
        """
        :param: json_file: json file to be validated
        :param: json_schema: json schema to validate the json_file against
        """
        self._json_file = file_spec
        self._json_object = None
        validate_file(file_spec)
        self.__validate_json_syntax(file_spec)
        self.__validate_json_schema(schema)

    @property
    def json_object(self):
        return self._json_object

    def __validate_json_syntax(self, file_spec):
        with open(file_spec) as json_file:
            try:
                self._json_object = json.load(json_file)
            except json.decoder.JSONDecodeError as ex:
                raise SyntaxError(f'syntax error in {self._json_file}')

    def __validate_json_schema(self, schema):
        try:
            validate(self._json_object, schema)
        except jsonschema.exceptions.ValidationError:
            raise jsonschema.exceptions.ValidationError(f'{self._json_file} does not conform to schema')


class MetricDefinitionParser(ABC):
    """
    Metric definition parser abstract base class (ABC)
    """

    retire_latency_str = 'retire_latency'
    metric_fields_class = None  # This will be set in subclasses

    def __init__(self, file_path: Path):
        """
        Constructor
        :param file_path: metric definition file to parse
        """
        self._metric_file = file_path

    def parse(self) -> List[MetricDefinition]:
        """
        Parse metric definitions
        :return: list of parsed metrics
        """
        validate_file(self._metric_file)
        return self._get_metric_definitions()

    def _get_metric_definitions(self):
        metrics = self._get_metrics()
        metric_defs = []
        for m in metrics:
            metric_defs.append(self._create(m))
        return metric_defs

    @property
    @abstractmethod
    def metric_fields_class(self):
        """
        Returns the class used to parse metric fields.
        This should be set in subclasses.
        """
        return self.metric_fields_class

    @abstractmethod
    def _get_metrics(self):
        pass

    def _create(self, metric_def):
        metric_fields = self._get_fields(metric_def)
        return MetricDefinition(*metric_fields)

    def _get_fields(self, metric_def):
        metric_fields = self.metric_fields_class(metric_def)
        return metric_fields.fields


class JsonParser(MetricDefinitionParser):
    """
    Parser for Data Lake metric definition files (JSON)
    """

    metric_fields_class = _JsonMetricFields

    def _get_metrics(self):
        with open(self._metric_file) as metrics_file:
            metric_defs_json = json.load(metrics_file)
        return metric_defs_json['Metrics']


class XmlParser(MetricDefinitionParser):
    """
    Parser for XML metric definition files
    """

    metric_fields_class = _XmlMetricFields

    def _get_metrics(self):
        root = ET.parse(self._metric_file, forbid_dtd=True, forbid_entities=True, forbid_external=True).getroot()
        return root.findall('metric')

    @staticmethod
    def _verify_constants_in_formula(name: str, constants: Dict[str,str], formula:str):
        for constant in constants:
            if constant not in formula:
                logging.debug(f'Constant \'{constant}\', defined for \'{name}\', is not used in the metric formula.')


class MetricDefinitionParserFactory:
    """
    Create a metric definition parser based on file type. Use `create(file_path)` method to create the appropriate
    metric definition parser.
    """
    parser_for_file_type = {
        '.xml': XmlParser,
        '.json': JsonParser
    }

    @classmethod
    def create(cls, file_path: Path):
        """
        Creates a parser object based on the given file type
        :param file_path: metric file to parse
        :return: an implementation of MetricDefinitionParser suitable for parsing the specified file
        """
        if file_path.suffix not in cls.parser_for_file_type:
            raise ValueError(f'No metric definition parser defined for files of type {file_path.suffix}')
        return cls.parser_for_file_type[file_path.suffix](file_path)


class JsonConstantParser:
    """
    Parser for json retire latency constant definition files
    """

    # retire latency schema for validating -l/--retire-latency json
    # command line input see SDL: T1148/T1149
    # https://sdp-prod.intel.com/bunits/intel/thor-next-gen-edp/thor/tasks/phase/development/11130-T1148/
    # https://sdp-prod.intel.com/bunits/intel/thor-next-gen-edp/thor/tasks/phase/development/11130-T1149/
    schema = {
        "type": "object",
        "properties": {
            "Data": {
                "type": "object",
                "patternProperties": {
                    ".*": {
                        "type": "object",
                        "properties": {
                            "MEAN": {
                                "type": "number",
                                "minimum": 0.0,
                                "maximum": 1000000.0
                            },
                            "other-stat": {
                                "type": "number",
                                "minimum": 0.0,
                                "maximum": 1000000.0
                            },
                        },
                        "required": ["MEAN"]
                    }
                }
            }
        },
        "required": ["Data"]
    }

    def __init__(self, file_path: Path, constant_descriptor: str):
        """
        :param file_path: file to parse metric constants from
        """
        self.__constant_descriptor = constant_descriptor
        self.__json_file = file_path

    def parse(self) -> Dict[str, float]:
        """
        Parse constants from json file
        :return: dictionary of parsed constants
        """
        json_object_validator = JsonObjectValidator(self.__json_file, JsonConstantParser.schema)
        json_constants = json_object_validator.json_object
        return self._get_constants_for_descriptor(json_constants)

    def _get_constants_for_descriptor(self, json_constants) -> Dict[str, float]:
        """
        Extract constants that match the constant descriptor from a json dictionary
        :param json_constants: dictionary of constants from parsed from json
        :return: dictionary of constants that match the constant descriptor or an
                 empty dictionary if the constant descriptor isn't found
        """
        constants = {}
        data_section = json_constants.get("Data", {})
        for c in data_section:
            if self.__constant_descriptor in data_section[c]:
                constant = self._get_retire_latency_constant_name(c)
                constants[constant] = data_section[c].get(self.__constant_descriptor)
        return constants

    @staticmethod
    def _get_retire_latency_constant_name(constant) -> str:
        if RETIRE_LATENCY_STRING not in constant:
            return constant + ':' + RETIRE_LATENCY_STRING
        return constant
