/*******************************************************************************
 * Copyright 2015 Intel Corporation.
 *
 *
 * This software and the related documents are Intel copyrighted materials, and your use of them is governed by
 * the express license under which they were provided to you ('License'). Unless the License provides otherwise,
 * you may not use, modify, copy, publish, distribute, disclose or transmit this software or the related
 * documents without Intel's prior written permission.
 * This software and the related documents are provided as is, with no express or implied warranties, other than
 * those that are expressly stated in the License.
 *******************************************************************************/

#ifndef __OWNISPLITL_H__
#define __OWNISPLITL_H__

#include "owndefs_tl.h"
#ifdef __cplusplus
extern "C" {
#endif

typedef enum {
    ippSplitAdd,
    ippSplitSquare
} SplitOperation;

/* ////////////////////////////////////////////////////////////////////////////
//  Names:      owniSplit_LT
//
//  Purpose:    Splits the integral parameter splitSize on tiles according the following condition
//
//              ((splitSize / numTiles) + (splitSize % numTiles)) * multiplier <= threshold
//
//  Parameters:
//   splitSize        Size to split
//   multiplier         Formula's multiplier
//   threshold          Formula's threshold
//   splitOp            Approach of tile's number increasing
//                          ippSplitAdd     - increase tile's number on 1
//                          ippSplitSquare  - increase tile's number in 2 times
//   tileLength         Tile's length (in-out parameter)
//   residual           Residual - tail (in-out parameter)
//   numTiles           Number of tiles (in-out parameter)
*/
void owniSplit_LT(IppSizeL splitSize, IppSizeL multiplier, IppSizeL threshold, SplitOperation splitOp, IppSizeL *tileLength, IppSizeL *residual,
                  IppSizeL *numTiles);

/* ////////////////////////////////////////////////////////////////////////////
//  Names:      owniSplitUniform2D
//
//  Purpose:    Splits the image on tiles
//
//  Parameters:
//   roiSize            Image size
//   pixelSize          Size of image pixels in bytes
//   minTileSize        Minimum size of tile in pixels
//   pSplit             Number of split parts along x and y axes
//   pTileSize          The size of a tile
//   pTailSize          The size of the last (corner) tile
*/
extern IppStatus owniSplitUniform2D_LT(IppiSizeL roiSize, IppSizeL minTileSize, IppiPointL *pSplit, IppiSizeL *pTileSize, IppiSizeL *pTailSize);

/* ////////////////////////////////////////////////////////////////////////////
//  Names:      owniGetTileParamsByIndex
//
//  Purpose:    Get offset and size of the tile by the given index
//
//  Parameters:
//   index              ordinal index of a tile
//   splitImage         Split of the image by x- and y- axis correspondingly
//   tileSize           size of a tile
//   tailSize           size of the last right-bottom tile
//   pTileOffset        offset of the tile corresponding the left-top image corner
//   pTileSize          size of the tile
*/
extern IppStatus owniGetTileParamsByIndex_LT(IppSizeL index, IppiPointL splitImage, IppiSizeL tileSize, IppiSizeL tailSize, IppiPointL *pTileOffset,
                                             IppiSizeL *pTileSize);

/* ////////////////////////////////////////////////////////////////////////////
//  Names:      owniGetTileSizeSimple_LT
//
//  Purpose:    Computes the tile size depending on input data
//
//  Parameters:
//   roiSize            Image size
//   minPixelNumber     Minimal number of pixels in the tile
//   minTileSize        Minimal size of tile along x and y axes
//   pTileSize          The size of a tile
//
*/
extern void owniGetTileSizeSimple_LT(IppiSizeL roiSize, IppSizeL minPixelNumber, IppiSizeL minTileSize, IppiSizeL *pTileSize);

/* ////////////////////////////////////////////////////////////////////////////
//  Names:      owniSplitToTiles_L
//
//  Purpose:    Splits the image on tiles with the predefined tile size
//
//  Parameters:
//   roiSize            Image size
//   tileSize           Predefined tile size, it can be zero, in this case the tile size is set by the function
//   pSplit             Number of split parts along x and y axes
//   pTileSize          The size of a tile
//   pLastSize          The size of the last (corner) tile
//
*/
extern void ippiSplitToTiles_LT(IppiSizeL roiSize, IppiSizeL tileSize, IppiPointL *pSplit, IppiSizeL *pTileSize, IppiSizeL *pLastSize);
/* ////////////////////////////////////////////////////////////////////////////
//  Names:      owniSplitToCubes_LT
//
//  Purpose:    Splits the volume on cubes with the predefined tile size
//
//  Parameters:
//   roiVolume          Image volume
//   cubeVolume         Predefined cube volume, it can be zero, in this case the cube volume is set by the function
//   pSplit             Number of split parts along x and y axes and z axes
//   pCubeVolume        The volume of a  cube
//   pLastVolume        The volume of the last (corner) cube
//
*/
extern void ipprSplitToCubes_LT(IpprVolumeL roiVolume, IpprVolumeL cubeVolume, IpprPointL *pSplit, IpprVolumeL *pCubeVolume,
                                IpprVolumeL *pLastVolume);

#ifdef __cplusplus
}
#endif

#endif /* __OWNISPLITL_H__ */
