package Langertha::Engine::vLLM;
# ABSTRACT: vLLM inference server
our $VERSION = '0.200';
use Moose;
use Carp qw( croak );

extends 'Langertha::Engine::OpenAI';


has '+url' => (
  required => 1,
);

sub default_model { croak "".(ref $_[0])." requires a default_model" }

sub _build_api_key { 'vllm' }

sub _build_supported_operations {[qw(
  createChatCompletion
  createCompletion
)]}


1;

__END__

=pod

=encoding UTF-8

=head1 NAME

Langertha::Engine::vLLM - vLLM inference server

=head1 VERSION

version 0.200

=head1 SYNOPSIS

    use Langertha::Engine::vLLM;

    my $vllm = Langertha::Engine::vLLM->new(
        url          => $ENV{VLLM_URL},
        model        => $ENV{VLLM_MODEL},
        system_prompt => 'You are a helpful assistant',
    );

    print $vllm->simple_chat('Say something nice');

    # MCP tool calling (requires server started with tool-call-parser)
    use Future::AsyncAwait;

    my $vllm = Langertha::Engine::vLLM->new(
        url         => 'http://localhost:8000/v1',
        model       => 'Qwen/Qwen2.5-3B-Instruct',
        mcp_servers => [$mcp],
    );

    my $response = await $vllm->chat_with_tools_f('Add 7 and 15');

=head1 DESCRIPTION

Provides access to vLLM, a high-throughput inference engine for large
language models. Extends L<Langertha::Engine::OpenAI> since vLLM exposes
an OpenAI-compatible API.

Both C<url> and C<model> are required. The URL must include the C</v1>
path prefix (e.g., C<http://localhost:8000/v1>). The API key defaults to
C<'vllm'> since local vLLM instances typically don't require authentication.

MCP tool calling requires the vLLM server to be started with
C<--enable-auto-tool-choice> and C<--tool-call-parser> matching the model
(C<hermes> for Qwen2.5/Hermes, C<llama3> for Llama, C<mistral> for Mistral).

See L<https://docs.vllm.ai/> for installation and configuration details.

B<THIS API IS WORK IN PROGRESS>

=head1 SEE ALSO

=over

=item * L<https://docs.vllm.ai/> - vLLM documentation

=item * L<Langertha::Engine::OpenAI> - Parent engine (OpenAI-compatible API)

=item * L<Langertha::Role::Tools> - MCP tool calling interface

=item * L<Langertha> - Main Langertha documentation

=back

=head1 SUPPORT

=head2 Issues

Please report bugs and feature requests on GitHub at
L<https://github.com/Getty/langertha/issues>.

=head1 CONTRIBUTING

Contributions are welcome! Please fork the repository and submit a pull request.

=head1 AUTHOR

Torsten Raudssus <torsten@raudssus.de> L<https://raudss.us/>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2026 by Torsten Raudssus.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
