/* BEGIN software license
 *
 * MsXpertSuite - mass spectrometry software suite
 * -----------------------------------------------
 * Copyright (C) 2009--2026 Filippo Rusconi
 *
 * http://www.msxpertsuite.org
 *
 * This file is part of the MsXpertSuite project.
 *
 * The MsXpertSuite project is the successor of the massXpert project. This
 * project now includes various independent modules:
 *
 * - massXpert, model polymer chemistries and simulate mass spectrometric data;
 * - mineXpert, a powerful TIC chromatogram/mass spectrum viewer/miner;
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 *
 * END software license
 */

#pragma once

/////////////////////// stdlib includes


/////////////////////// Qt includes
#include <QList>
#include <QMap>
#include <QFileInfo>


/////////////////////// pappsomspp includes
#include "pappsomspp/core/trace/trace.h"


/////////////////////// Local includes
#include "MsXpS/export-import-config.h"
#include "MsXpS/libXpertMassCore/Formula.hpp"
#include "MsXpS/libXpertMassCore/IsotopicData.hpp"
#include "MsXpS/libXpertMassCore/IsotopicDataLibraryHandler.hpp"

namespace MsXpS
{
namespace libXpertMassCore
{


// National Institute of Standards and Technology - CODATA 2018
// 2018, Reviews of Modern Physics
static constexpr double PROTON_MASS           = 1.007276466621;
// A neutron not engaged in the formation of an atom.
static constexpr double NEUTRON_MASS          = 1.00866491606;
// A neutron participating into an atom (mass defect for Carbon).
static constexpr double ISOTOPIC_NEUTRON_MASS = 1.0033548378;

struct DECLSPEC SupportingIon
{
  int charge;            // z = 1..3
  double monoisotopicMz; // observed monoisotopic m/z
  double intensity;      // observed intensity at monoisotopic peak
};

struct DECLSPEC IsoMatch
{
  size_t theoreticalClusterIndex;
  size_t observedPeakIndex;
  double observedMz;
  double observedIntensity;
};

struct BinnedIsotopicCentroid
{
  int isotopeIndex;   // 0 = mono, 1 = M+1, ...
  double neutralMass; // weighted centroid
  double intensity;   // summed intensity

  QString
  toString() const
  {
    QString text;

    text += QString(
              "Isotopic centroid index: %1 - "
              "Neutral mass: %2 "
              "Intensity: %3\n")
              .arg(isotopeIndex)
              .arg(neutralMass, 0, 'f', 5)
              .arg(intensity);

    return text;
  }
};

struct DECLSPEC FitQuality
{
  double correlation;        // Pearson r (shape agreement)
  double spacingErrorPpm;    // mean or RMS isotope spacing error
  double explainedIntensity; // fraction of observed intensity explained
};

struct DECLSPEC DeconvolutedFeature
{
  double neutralMass = 0; // neutral monoisotopic mass
  double intensity   = 0; // summed monoisotopic intensity
                          // across all supporting charge states

  std::vector<SupportingIon> supportingIons;

  FitQuality fitQuality;

  DeconvolutedFeature(): neutralMass(0), intensity(0) {};

  DeconvolutedFeature(const DeconvolutedFeature &other)
    : neutralMass(other.neutralMass), intensity(other.intensity)
  {
    for(const SupportingIon &supp_ion : other.supportingIons)
      supportingIons.push_back(
        {supp_ion.charge, supp_ion.monoisotopicMz, supp_ion.intensity});

    fitQuality.correlation        = other.fitQuality.correlation;
    fitQuality.spacingErrorPpm    = other.fitQuality.spacingErrorPpm;
    fitQuality.explainedIntensity = other.fitQuality.explainedIntensity;
  }

  QString
  toString(bool with_supporting_ions = true) const

  {
    QString text = QString("Neutral mass: %1, intensity: %2")
                     .arg(neutralMass, 0, 'f', 6)
                     .arg(intensity);

    if(with_supporting_ions)
      {
        if(supportingIons.size() == 1)
          text += "\nOne supporting ion\n";
        else
          text += QString("\nSupporting ions: %1\n").arg(supportingIons.size());

        for(const SupportingIon &supp_ion : supportingIons)
          text += QString("Mono m/z: %1 - z: %2, intensity: %3\n")
                    .arg(supp_ion.monoisotopicMz, 0, 'f', 6)
                    .arg(supp_ion.charge)
                    .arg(supp_ion.intensity);
      }

    text += with_supporting_ions ? "" : "\n";

    text += QString(
              "Fit quality: correlation: %1 - spacing error (ppm): %2 - "
              "explained intensity: %3\n")
              .arg(fitQuality.correlation)
              .arg(fitQuality.spacingErrorPpm)
              .arg(fitQuality.explainedIntensity);

    return text;
  }
};

struct DECLSPEC ChargeSpecificFeature
{
  int charge;
  double neutralMass;
  double monoisotopicMz;
  double intensity;

  FitQuality fitQuality;

  QString
  toString(bool with_fit_quality = false) const
  {
    QString text =
      QString("charge: %1 - neutral mass: %2 - mono m/z: %3 - intensity: %4\n")
        .arg(charge)
        .arg(neutralMass, 0, 'f', 5)
        .arg(monoisotopicMz, 0, 'f', 5)
        .arg(intensity);

    if(with_fit_quality)
      {
        text += "\nFit quality:\n";

        text += QString(
                  "Fit quality: correlation: %1 - spacing error (ppm): %2 - "
                  "explained intensity: %3\n")
                  .arg(fitQuality.correlation)
                  .arg(fitQuality.spacingErrorPpm)
                  .arg(fitQuality.explainedIntensity);
      }
    return text;
  }
};

////////////////////////// LowMassDeconvolver /////////////////////////
////////////////////////// LowMassDeconvolver /////////////////////////
////////////////////////// LowMassDeconvolver /////////////////////////
////////////////////////// LowMassDeconvolver /////////////////////////
////////////////////////// LowMassDeconvolver /////////////////////////

// This class processes a centroided Trace and detects the isotopic clusters.
// For each detected cluster, it stores the monoisotopic m/z and the z.

class DECLSPEC LowMassDeconvolver
{

  public:
  struct Parameters
  {
    // Charge settings
    int minCharge = 1;
    int maxCharge = 3;

    // Mass range (for low mass)
    double minMass = 200.0;  // Da
    double maxMass = 3500.0; // Da

    // Mass accuracy
    double ppmMassTolerance = 5.0; // tight tolerance for high-res
    // instruments

    // Intensity filtering
    double minIntensity = 100.0;

    // The isotopic data file path for the data required for the averagine-based
    // computations
    QString isotopicDataFilePath;

    // For real work.
    IsotopicDataSPtr isotopicDataSp;

    // Averagine formula, for Ecoli muropeptides: "C4.29H6.86N0.86O2.00"
    Formula averagineFormula;

    // Monoisotopic mass of one averagine unit (i.e. scaling reference)
    double averagineMonoMass = 0;

    // When comparing the relative intensity of the observed vs theoretical
    // isotopic cluster centroids, two-by-tow (isotopologues 0, then
    // isotopologues 1), the following value is the tolerance that is accepted
    // for a deviation in the observed centroid intensity with respect to the
    // theoretical intensty. For example 0.6 would mean: "tolerate that the
    // intensity difference between observed and theoretical isotopologues at
    // position 0 of the cluster be up to (0.6 * theoretical intensity):
    // if (abs(obs_intensity - theo_intensity) > 0.6 * theoretical intensity ->
    // REJECT.
    double clusterShapeTolerance = 0.7;

    // Scoring threshold (optional)
    double minScore = 0.3;

    Parameters() {};

    Parameters(int min_charge,
               int max_charge,
               double min_mass,
               double max_mass,
               double ppm_mass_tolerance,
               double min_intensity,
               const QString &isotopic_data_file_path,
               const QString &averagine_formula_string,
               double relative_cluster_shape_tolerance,
               double min_score)
      : minCharge(min_charge),
        maxCharge(max_charge),
        minMass(min_mass),
        maxMass(max_mass),
        ppmMassTolerance(ppm_mass_tolerance),
        minIntensity(min_intensity),
        isotopicDataFilePath(isotopic_data_file_path),
        averagineFormula(averagine_formula_string),
        clusterShapeTolerance(relative_cluster_shape_tolerance),
        minScore(min_score)
    {
      Q_ASSERT(!isotopicDataFilePath.isEmpty());
      Q_ASSERT(QFileInfo::exists(isotopicDataFilePath));

      // Load the isotopic data for all the isotopic cluster simulations in
      // LowMassDeconvolver.
      IsotopicDataLibraryHandler isotopic_data_handler;
      qsizetype loaded_isotope_count =
        isotopic_data_handler.loadData(isotopicDataFilePath);
      Q_ASSERT(loaded_isotope_count);

      isotopicDataSp = isotopic_data_handler.getIsotopicData();
      Q_ASSERT(isotopicDataSp != nullptr && isotopicDataSp.get() != nullptr);

      Q_ASSERT(
        !averagineFormula.getActionFormula(false /*withTitle*/).isEmpty());
      Formula averagine_formula(averagineFormula);
      qDebug() << "Initializing averagine formula with:"
               << averagineFormula.getActionFormula(false /*withTitle*/);

      ErrorList error_list;
      if(!averagineFormula.validate(isotopicDataSp, &error_list))
        qFatal() << "Failed to validate the averagine formula, with errors:"
                 << Utils::joinErrorList(error_list);

      // Initialize averagineMonoMass
      averagineMonoMass = 0;
      bool ok           = false;
      double avg;
      averagineFormula.accountMasses(
        ok, isotopicDataSp, averagineMonoMass, avg, 1);
      Q_ASSERT(ok);
      qDebug() << "Computed averagine" << averagineFormula.getActionFormula()
               << "mono mass:" << averagineMonoMass;
    }

    Parameters(const Parameters &other)
    {
      *this = other;
    }

    Parameters &
    operator=(const Parameters &other)
    {
      if(&other == this)
        return *this;

      minCharge             = other.minCharge;
      maxCharge             = other.maxCharge;
      minMass               = other.minMass;
      maxMass               = other.maxMass;
      ppmMassTolerance      = other.ppmMassTolerance;
      minIntensity          = other.minIntensity;
      isotopicDataFilePath  = other.isotopicDataFilePath;
      isotopicDataSp        = other.isotopicDataSp;
      averagineFormula      = other.averagineFormula;
      averagineMonoMass     = other.averagineMonoMass;
      clusterShapeTolerance = other.clusterShapeTolerance;
      minScore              = other.minScore;
      return *this;
    }

    void
    initialize(int min_charge,
               int max_charge,
               double min_mass,
               double max_mass,
               double ppm_mass_tolerance,
               double min_intensity,
               const QString &isotopic_data_file_path,
               const QString &averagine_formula_string,
               int relative_cluster_shape_tolerance,
               double min_score)
    {
      minCharge             = min_charge;
      maxCharge             = max_charge;
      minMass               = min_mass;
      maxMass               = max_mass;
      ppmMassTolerance      = ppm_mass_tolerance;
      minIntensity          = min_intensity;
      isotopicDataFilePath  = isotopic_data_file_path;
      averagineFormula      = Formula(averagine_formula_string);
      clusterShapeTolerance = relative_cluster_shape_tolerance;
      minScore              = min_score;

      Q_ASSERT(!isotopicDataFilePath.isEmpty());
      Q_ASSERT(QFileInfo::exists(isotopicDataFilePath));

      // Load the isotopic data for all the isotopic cluster simulations in
      // LowMassDeconvolver.
      IsotopicDataLibraryHandler isotopic_data_handler;
      qsizetype loaded_isotope_count =
        isotopic_data_handler.loadData(isotopicDataFilePath);
      Q_ASSERT(loaded_isotope_count);

      isotopicDataSp = isotopic_data_handler.getIsotopicData();
      Q_ASSERT(isotopicDataSp != nullptr && isotopicDataSp.get() != nullptr);

      Q_ASSERT(
        !averagineFormula.getActionFormula(false /*withTitle*/).isEmpty());
      Formula averagine_formula(averagineFormula);
      qDebug() << "Initializing averagine formula with:"
               << averagineFormula.getActionFormula(false /*withTitle*/);

      ErrorList error_list;
      if(!averagineFormula.validate(isotopicDataSp, &error_list))
        qFatal() << "Failed to validate the averagine formula, with errors:"
                 << Utils::joinErrorList(error_list);

      // Initialize averagineMonoMass
      averagineMonoMass = 0;
      bool ok           = false;
      double avg;
      averagineFormula.accountMasses(
        ok, isotopicDataSp, averagineMonoMass, avg, 1);
      Q_ASSERT(ok);
      qDebug() << "Computed averagine" << averagineFormula.getActionFormula()
               << "mono mass:" << averagineMonoMass;
    }

    QString
    toString() const
    {
      QString text;
      text += QString(
                "\nmin charge: %1 - "
                "max charge: %2 - "
                "min mass: %3 - "
                "max mass: %4 - "
                "mass tolerance (ppm): %5 - "
                "min intensity: %6 - "
                "isotopic data file path: '%7' - "
                "averagine formula: '%8' - "
                "averagine mono mass: %9 - "
                "min fit percentage for isotopic cluster shape: %10 - "
                "min score: %11\n")
                .arg(minCharge)
                .arg(maxCharge)
                .arg(minMass)
                .arg(maxMass)
                .arg(ppmMassTolerance, 0, 'f', 5)
                .arg(minIntensity)
                .arg(isotopicDataFilePath)
                .arg(averagineFormula.getActionFormula())
                .arg(averagineMonoMass, 0, 'f', 5)
                .arg(clusterShapeTolerance)
                .arg(minScore);
      return text;
    }
  };

  explicit LowMassDeconvolver(Parameters &parameters);

  ~LowMassDeconvolver();

  std::vector<DeconvolutedFeature>
  deconvolute(const pappso::Trace &centroided_mass_spectrum) const;

  protected:
  Parameters m_params;

  inline double
  neutralToMz(double neutral_mass, int charge) const
  {
    return (neutral_mass + charge * PROTON_MASS) / charge;
  }

  inline double
  mzToNeutral(double mz, int charge) const
  {
    return mz * charge - charge * PROTON_MASS;
  }

  inline double
  ppmToDelta(double mass_or_mz, double ppm) const
  {
    return (mass_or_mz * ppm) / 1e6;
  }

  inline double
  deltaToPpm(double delta_mass_or_mz, double mass_or_mz) const
  {
    // ppm = (|observed - theoretical| × 1,000,000) / theoretical
    return delta_mass_or_mz * 1e6 / mass_or_mz;
  }

  inline bool
  neutralMassesMatch(double m1, double m2) const
  {
    return std::abs(m1 - m2) <= m_params.ppmMassTolerance;
  }

  QString compositionEstimationFromMass(double neutral_mass) const;

  pappso::Trace isotopicClusterForFormula(const QString &formula) const;

  pappso::Trace isotopicClusterForNeutralMass(double neutral_mass) const;

  std::vector<BinnedIsotopicCentroid> collapseTheoreticalIsotopicCluster(
    const pappso::Trace &theoretical_cluster) const;

  std::vector<pappso::DataPoint>
  selectCandidatePeaks(const pappso::Trace &input_centroids) const;

  bool fitIsotopicEnvelope(const pappso::Trace &input_centroids,
                           std::size_t input_centroid_index,
                           ChargeSpecificFeature &feature,
                           std::vector<bool> &globally_used) const;

  std::vector<ChargeSpecificFeature>
  generateChargeSpecificFeatures(const pappso::Trace &input_centroids) const;

  std::vector<DeconvolutedFeature> groupByNeutralMass(
    const std::vector<ChargeSpecificFeature> &charge_specific_features) const;

  void resolveOverlappingEnvelopes(
    std::vector<ChargeSpecificFeature> &features) const;

  double
  computeIsotopeSpacingPpm(const std::vector<pappso::DataPoint> &matched_peaks,
                           int charge) const;
};

} // namespace libXpertMassCore
} // namespace MsXpS
